function [  ] = main()
%Main-file for calculating external stimuli for a regulatory network
%causing a switch between two steady states of the regulatory network
%For details see corresponding paper
%Date Dec 21, 2017

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Parameters determined by the network
%numNodes:     	Number of nodes of the network 
%numControls:  	Number of controls of the network
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Parameters determined by the user
%timeInterval:  Number, time discretization for the ordinary differential equations
%timeHorizon:   Number, time duration, [0, timeHorizon], in which network is simulated
%alpha:         Number, Weights the contribution of the controls to the target functional
%initialState:  1 x numNodes row vector, The steady state the network is in at the beginning, that is time t=0
%finalState:    1 x numNodes row vector, The steady state in which the network is expected to be at the latest at the final time, t=timeHorizon

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%tol1=10^-1;         %Tolerance for |x(i)-xd(i)|<tol1 for i=1,...,numNodes at the final time timeHorizon, input for the function combinatorial_method
%tol2=10^-6;         %Tolerance for the stopping criterion for the sequential quadratic Hamiltonian method, input in function SQH_method
%tol3=10^-4;         %Tolerance for the stopping criterion for the projected gradient method, input in function projected_gradient_method
%T_int=10^-1;        %Tolerance to determine smallest duration of application of external stimuli causing a switch, combinatorial_method         
%max_Num=4;          %Input in function combinatorial_method, Maximum number of external stimuli applied at once to the network in order to find a set of external stimuli that cause the desired switch, max_Num = 1,...,numControls
%max_iter=10000;     %Maximum number of updates on the control for the sequential quadratic Hamiltonian method or maximum number of iterations of the projected gradient method

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%flags
%combi_method=1;                 %If flag equals 1, a combinatorial search with the function combinatorial_method by trial and error search is performed in ordert to determine external stimuli causing the desired switch, if flag equals 0 it is not performed
%local_optimization_method=1;    %If flag equals 0, no local optimization method is performed, if local_optimization_method equals 1, then the sequential quadratic Hamiltonian (SQH) method is performed, if local_optimization_method equals 2, then a projected gradient method is performed

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Variables
%x:         numNodes x ((timeHorizon/timeInterval)+1)-matrix, state of the network, row i corresponds 
%           to the  node i, i=1,...,numNodes, column j corresponds to
%           the time t=(j-1)*timeInterval, j=1,...,((timeHorizon/timeInterval)+1), entry (i,j) corresponds to value x(i) of the i-th
%           node at time t=(j-1)*timeInterval
%xd:        numNodes x ((timeHorizon/timeInterval)+1)-matrix, Desired state
%           of the netword at final time timeHorizon, row i corresponds to the desired
%           state of node i, i=1,...,numNodes, column j corresponds to the time 
%           t=(j-1)*timeInterval, j=1,...,((timeHorizon/timeInterval)+1), entry (i,j) value of the i-th
%           desired state xd(i) of node i at time t=(j-1)*timeInterval
%u:         numControls x (timeHorizon/timeInterval)-matrix, external stimuli, row
%           i corresponds to the external stimulus u(i), i=1,...,numControls, column j
%           corresponds to the time t=(j-1)*timeInterval,
%           j=1,...,(timeHorizon/timeInterval), entry (i,j) value of the i-th
%           external stimulus u(i) at time t=(j-1)*timeInterval
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

tol1=10^-1;
tol2=10^-6;
tol3=10^-4;
T_int=10^-1;
max_Num=4;
max_iter=10000;
combi_method=1;
local_optimization_method=1;

OCP=struct('numNodes',4,'numControls',4,'timeInterval',0.1,'timeHorizon',50.0,'alpha',0.1,'initialState',[0.8882285370369972,0.0034904637320149904,0.8882285364810297,0.003490463395483468]);

xd=get_xd([0.0034914728784034136,0.8882149249345953,0.003491473479228141,0.8882149253311926], OCP);
f= {@(x,u)((-exp(5.0)+exp(-10.0*(((3.0/2.0)*((x(2)+x(3))/(1+x(2)+x(3)))*(1-(11.0/10.0)*((10.0*x(4))/(1+10.0*x(4)))))-0.5)))/((1-exp(5.0))*(1+exp(-10.0*(((3.0/2.0)*((x(2)+x(3))/(1+x(2)+x(3)))*(1-(11.0/10.0)*((10.0*x(4))/(1+10.0*x(4)))))-0.5)))))-x(1)+u(1)*(1-x(1)),...
@(x,u)((-exp(5.0)+exp(-10.0*(((3.0/2.0)*((x(3)+x(2))/(1+x(3)+x(2)))*(1-(11.0/10.0)*((10.0*x(1))/(1+10.0*x(1)))))-0.5)))/((1-exp(5.0))*(1+exp(-10.0*(((3.0/2.0)*((x(3)+x(2))/(1+x(3)+x(2)))*(1-(11.0/10.0)*((10.0*x(1))/(1+10.0*x(1)))))-0.5)))))-x(2)+u(2)*(1-x(2)),...
@(x,u)((-exp(5.0)+exp(-10.0*(((3.0/2.0)*((x(3)+x(2))/(1+x(3)+x(2)))*(1-(11.0/10.0)*((10.0*x(4))/(1+10.0*x(4)))))-0.5)))/((1-exp(5.0))*(1+exp(-10.0*(((3.0/2.0)*((x(3)+x(2))/(1+x(3)+x(2)))*(1-(11.0/10.0)*((10.0*x(4))/(1+10.0*x(4)))))-0.5)))))-x(3)-u(4)*x(3),...
@(x,u)((-exp(5.0)+exp(-10.0*(((3.0/2.0)*((x(3)+x(2))/(1+x(3)+x(2)))*(1-(11.0/10.0)*((10.0*x(1))/(1+10.0*x(1)))))-0.5)))/((1-exp(5.0))*(1+exp(-10.0*(((3.0/2.0)*((x(3)+x(2))/(1+x(3)+x(2)))*(1-(11.0/10.0)*((10.0*x(1))/(1+10.0*x(1)))))-0.5)))))-x(4)-u(3)*x(4)};

u=zeros(OCP.numControls,round(OCP.timeHorizon/OCP.timeInterval));  %Initial guess for the controls if no combinatorial search is performed before the local optimization framework, any control can be set to any value between 0 and 1 for an initial guess for example ones() instead of zeros()

if(combi_method==1)                                         %Block for the combinatorial method
    fprintf('\n');
    fprintf('Starting combinatorial method...\n');
    A=combinatorial_method(f,xd,tol1,max_Num,T_int,OCP);    %Trys to get external stimuli causing desired switch by trial an error for different combinations of external stimuli, Output: array, left column number of stimuli, right column duration of its application from t=0
    if(isempty(A)==0)                                       %Check if any external stimuli have been found
        u=setControls(A,OCP);                               %Sets u according to the external stimuli returned by function combinatorial_method if the set of external stimuli returned by function combinatorial_method is not empty
        fprintf('\n');                                        
        fprintf('External stimuli from combinatorial method causing desired switch at a tolerance |x-xd|<%d:\n',tol1)       
        [rowA,~]=size(A);                                   %Number of the set of external stimuli which have been found causing the desired switch                                                                                     %Determine rows of A     
        for i=1:rowA                                                                              %Read out each row
            fprintf('External stimulus %i applied at t=0 for %d time units\n', A(i,1),A(i,2));    %Output results of combinatorial_method, that means read out matrix A if not empty
        end
    end
end

if(local_optimization_method==1 || local_optimization_method==2)                             %Block for the local optimization method                                                         
    [df_x,cmx,df_u,cmu]=createJacobian(f,OCP);                                               %Creates derivatives of f with respect to x and u, Jacobian of the right hand side f
    if(local_optimization_method==1)
        u=SQH_method( @get_J_SQH,f,df_x,cmx,df_u,cmu,tol2,u,xd,max_iter,OCP); %Sequential quadratic Hamiltonian method as a local optimization scheme, returns u, u the external stimuli optimizing the target functional
                                                                              %Input: @get_J function handle for the target functional, f right hand-side of the ordinary differential equation corresponding to the network with dx/dt=f(x(t),u(t))
                                                                              %df_x function handle for the derivative of f with respect to x, cmx notes the nonzero elements of df_x, df_u function handle for the derivative of f with respect to u, cmu notes the nonzero elements of df_u, see output function createJacobian
                                                                              %u inital guess for the external stimuli, can be taken from the combinatorial method 
                                                                              %xd desired state for the values x of the corresponding nodes, tol2 stopping criterion, max_iter maximum number of updates on the control u of the sequential quadratic Hamiltonian
    end                                                              
                                                                                                                                                            
    if (local_optimization_method==2)
        [u,~,~]=projected_gradient_method( @get_gradient, @projection, @get_J,f,df_x,cmx,df_u,cmu, u, xd, tol3, max_iter, OCP );    %Prjected gradient method as a local optimization scheme, returns [u,J,count], u the external stimuli optimizing the target functional
                                                                                                                                    %Input:@get_gradient function handle for the gradient of the reduced target funcitonal, @projection function handel of the projection, projects u into [0,1] 
                                                                                                                                    %@get_J function handle for the target functional, f right hand-side of the ordinary differential equation corresponding to the network with dx/dt=f(x(t),u(t))
                                                                                                                                    %df_x function handle for the derivative of f with respect to x, cmx notes the nonzero elements of df_x, df_u function handle for the derivative of f with respect to u, cmu notes the nonzero elements of df_u, see output function createJacobian
                                                                                                                                    %u inital guess for the external stimuli, can be taken from the combinatorial method 
                                                                                                                                    %xd desired state for the values x of the corresponding nodes, tol2 stopping criterion, max_iter maximum iteration number of the projected gradient method  
    end
end

if(max(max(u))~=0)                          %Print the numbers of active external stimuli if there is an external stimulus being different from a constant zero function
    fprintf('\n');
    fprintf('Active external stimuli, that means being different from constant zero function:\n');                      
    for i=1:OCP.numControls                 %Prints out the numbers which correspond to the active external stimuli, can be copied for the function drawStimuli  
        if(max(u(i,:))~=0)
            fprintf('%i,',i);
        end
    end
    fprintf('\n');
else
    fprintf('No active external stimulus\n');
end

x=forward(f,u,OCP);                           %Calculates the state of the network corresponding to the external stimuli u calculated with the schemes above

fprintf('\n');
fprintf('Final values of the state x at final time T\n');         
fprintf('Number node\t Value node\n')
for i=1:OCP.numNodes                          %Output of the value of the state x at final time T
    fprintf('%i\t\t %d\n',i,x(i,end));
end

if(min((abs(x(:,end)-xd(:,end))<tol1)))       %Checks if final external stimuli cause the desired switch up to tolerance tol1
    fprintf('\n');
    fprintf('Each node equals its desired value at final time within its tolerance, that means |x_i(T)-x_d_i|<%d\n',tol1);
else
    fprintf('\n');
    fprintf('There is a node that does not have its desired value at the end, that means |x_i(T)-x_d_i|>=%d\n',tol1);
end

fprintf('\n');
fprintf('Save data to file...\n');
dlmwrite('x.txt',[0:OCP.timeInterval:round(OCP.timeHorizon/OCP.timeInterval)*OCP.timeInterval;x]);               %Writes the state x in a text-file "x.txt" where the first row corresponds to the discrete time steps, separated by commas, row i=2,...,numNode+1 corresponds to state x(i-1), values in the colums, separated by commas, value of x(i) at the corresponding time step  
dlmwrite('u.txt',[0:OCP.timeInterval:(round(OCP.timeHorizon/OCP.timeInterval)-1)*OCP.timeInterval;u]);           %Writes the external stimuli u in a text-file "u.txt" where the first row corresponds to the discrete time steps, separated by commas, row i=2,...,numControls+1 corresponds to external stimlus u(i), values in the colums, separated by commas, value of u(i) at the corresponding time step
fprintf('Done!\n')
end



%%%%%% Network information %%%%%%

 
% Node_list 
% Node 1 : CSF
% Node 2 : IL-2
% Node 3 : IL-4
% Node 4 : IFN
%
% Regulation_list
% Pos 1 : CSF
% Pos 2 : IL-2
% Neg 3 : IFN
% Neg 4 : IL-4
