import java.io.File;
import java.util.Arrays;
import java.util.HashMap;

import jimena.analysis.FunctionsAnalysis;
import jimena.binaryrn.RegulatoryNetwork;
import jimena.calculationparameters.StandardConvergenceParameters;
import jimena.customfunction.CustomFunction;
import jimena.simulation.BasicCalculationController;
import jimena.simulationmethods.SQUADMethod;

public class ExampleCustomFunction {

    public static void main(String[] args) throws Exception {
        // Load a yEd graph, in this case consisting of four nodes "A", "B", "C" and "D"
        RegulatoryNetwork network = new RegulatoryNetwork();
        network.loadYEdFile(new File("C:\\Users\\Stefan\\Desktop\\testnet.graphml"));

        // Set all nodes to the value 1
        network.getNodeByName("A").setValue(1.0);
        network.getNodeByName("B").setValue(1.0);
        network.getNodeByName("C").setValue(1.0);
        network.getNodeByName("D").setValue(1.0);

        // Start a simulation for 10 time units starting from the given network state [1,1,1,1]
        network.simulate(new SQUADMethod(), 0.0001D, 10D, Double.POSITIVE_INFINITY, 1.0D, new BasicCalculationController());
        System.out.println("Result of the simulation of the unaltered network: " + network);

        // Calculate the centralities in the network

        double[] centralities = FunctionsAnalysis.totalCentralities(network, 500,
                new StandardConvergenceParameters(new SQUADMethod(), 0.02));

        System.out.println("Centralities of the unaltered network: " + Arrays.toString(centralities));

        // Reset the network to the state [1,1,1,1]
        network.getNodeByName("A").setValue(1.0);
        network.getNodeByName("B").setValue(1.0);
        network.getNodeByName("C").setValue(1.0);
        network.getNodeByName("D").setValue(1.0);

        // Add a custom function for a node, in this case derivative(C) = A * C
        CustomFunction customFunctionNodeC = new CustomFunction() {
            @Override
            public double eval(HashMap<String, Double> inputs) {
                // The derivative of the value of C is now equal to the value of A time the value of C.
                return inputs.get("A") * inputs.get("C");

                // Notice that you can use the value of *any* node here, not just B and C which are the
                // inputs to node C in the original network.

                // other example:
                // return (Math.exp(inputs.get("A") * inputs.get("C"))+inputs.get("B")*inputs.get("D"));
                // ==> the node uses the values of all nodes in the network
            }
        };
        network.getNodeByName("C").setCustomFunction(customFunctionNodeC);

        // Simulate the network again, the results a expectedly different now
        network.simulate(new SQUADMethod(), 0.0001D, 10D, Double.POSITIVE_INFINITY, 1.0D, new BasicCalculationController());
        System.out.println("Result of the simulation of the altered network: " + network);

        // The centralities have changes as well
        centralities = FunctionsAnalysis.totalCentralities(network, 500, new StandardConvergenceParameters(new SQUADMethod(), 0.02));
        System.out.println("Centralities of the altered network: " + Arrays.toString(centralities));

    }

}
