package jimena.analysis;

import java.util.ArrayList;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import jimena.binaryrn.Connection;
import jimena.binaryrn.NetworkConnection;
import jimena.binaryrn.RegulatoryNetwork;
import jimena.calculationparameters.ConvergenceParameters;
import jimena.gui.main.JimenaExecutor;
import jimena.libs.ArrayLib;
import jimena.libs.BooleanValue;
import jimena.libs.MathLib;
import jimena.simulation.CalculationController;
import jimena.simulation.ConvergenceResult;
import jimena.simulation.StableSteadyState;

/**
 * A class providing methods to analyze the function of network nodes.
 *
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 *
 */
public class FunctionsAnalysis {

    /**
     * Auxiliary class to compute the minimum in the formula for dynamic centrality.
     *
     * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
     *
     */
    private class MinInfluence implements Callable<Double> {
        private RegulatoryNetwork network;
        private NetworkConnection connection = null;
        private int node = -1;
        private Set<Integer> nodesToConsider;
        private Set<Integer> protectedNodes;
        private ConvergenceParameters parameters;
        private int simulations;
        private BooleanValue aborted = new BooleanValue(false);

        public MinInfluence(RegulatoryNetwork network, int node, ConvergenceParameters parameters, Set<Integer> nodesToConsider,
                int simulations, Set<Integer> protectedNodes) {
            this.network = network;
            this.node = node;
            this.nodesToConsider = nodesToConsider;
            this.parameters = parameters;
            this.simulations = simulations;
            this.protectedNodes = protectedNodes;
        }

        public MinInfluence(RegulatoryNetwork network, NetworkConnection connection, ConvergenceParameters parameters,
                Set<Integer> nodesToConsider, int simulations, Set<Integer> protectedNodes) {
            this.network = network;
            this.connection = connection;
            this.nodesToConsider = nodesToConsider;
            this.parameters = parameters;
            this.simulations = simulations;
            this.protectedNodes = protectedNodes;
        }

        @Override
        public Double call() throws Exception {
            if (aborted.getValue()) {
                return Double.NaN;
            }

            double[] initialVector = MathLib.randomVector(network.size());
            if (connection != null) {
                return minInfluence(network, connection, initialVector, simulations, parameters, nodesToConsider, protectedNodes);
            } else {
                return minInfluence(network, node, initialVector, simulations, parameters, nodesToConsider, protectedNodes);
            }
        }
    }

    /**
     * Aborts all minimum calculations.
     *
     * @param runs
     *            The minimum calculations to abort.
     */
    private static void abortAll(ArrayList<Future<Double>> runs) {
        for (Future<Double> run : runs) {
            run.cancel(true); // TODO does this work?
        }
    }

    /**
     * Returns all sensitivities from an influence matrix.
     *
     * @param matrix
     *            The influence matrix to analyze.
     * @return A vector of all sensitivities.
     */
    public static double[] centralitiesFromMatrix(double[][] matrix) {
        if (matrix == null) {
            return null;
        }
        return centralitiesFromMatrix(matrix, MathLib.numberSet(matrix.length - 1));
    }

    /**
     * Returns the centralities from an influence matrix.
     *
     * @param matrix
     *            The influence matrix to analyze.
     * @param nodesToConsider
     *            The significant nodes.
     * @return A vector of all centralities
     */
    public static double[] centralitiesFromMatrix(double[][] matrix, Set<Integer> nodesToConsider) {
        if (matrix == null) {
            return null;
        }

        double[] result = new double[matrix.length];
        for (int i = 0; i < matrix.length; i++) {
            result[i] = centralityFromMatrix(matrix, i, nodesToConsider);
        }
        return result;
    }

    /**
     * Returns a centrality from an influence matrix.
     *
     * @param matrix
     *            The influence matrix to analyze.
     * @param node
     *            The node whose influence is to be returned.
     * @return Centrality of the node.
     */
    public static double centralityFromMatrix(double[][] matrix, int node) {
        return centralityFromMatrix(matrix, node, MathLib.numberSet(matrix.length - 1));
    }

    /**
     * Returns a centrality from an influence matrix.
     *
     * @param matrix
     *            The influence matrix to analyze.
     * @param node
     *            The node whose influence is to be returned.
     * @param nodesToConsider
     *            The significant nodes.
     * @return Centrality of the node.
     */
    public static double centralityFromMatrix(double[][] matrix, int node, Set<Integer> nodesToConsider) {
        if (matrix == null) {
            return Double.NaN;
        }

        double result = 0;
        int counter = 0;
        for (Integer i : nodesToConsider) {
            result += matrix[node][i];
            counter++;
        }
        return result / counter;
    }

    /**
     *
     * Returns the dynamic centralities of all nodes.
     *
     * @param network
     *            The network to analyze.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param innersimulations
     *            The number of simulations to approximate the minimum.
     * @param parameters
     *            Simulation parameters.
     * @return Dynamic centralities of all nodes.
     * @throws Exception
     */
    public static double[] dynamicCentralities(RegulatoryNetwork network, int simulations, int innersimulations,
            ConvergenceParameters parameters) throws Exception {
        return dynamicCentralities(network, simulations, innersimulations, parameters, network.setOfAllNodeIndices());
    }

    /**
     *
     * Returns the dynamic centralities of all nodes.
     *
     * @param network
     *            The network to analyze.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param innersimulations
     *            The number of simulations to approximate the minimum.
     * @param parameters
     *            Simulation parameters.
     * @param nodesToConsider
     *            The significant nodes.
     * @return Dynamic centralities of all nodes.
     * @throws Exception
     */
    public static double[] dynamicCentralities(RegulatoryNetwork network, int simulations, int innersimulations,
            ConvergenceParameters parameters, Set<Integer> nodesToConsider) throws Exception {
        double[] influences = new double[network.size()];
        for (int i = 0; i < network.size(); i++) {

            influences[i] = dynamicCentrality(network, i, simulations, innersimulations, parameters.cloneWithoutController(),
                    nodesToConsider);
            if (parameters.getCalculationController() != null && !parameters.getCalculationController().isOn()) {
                return null;
            }
            if (parameters.getCalculationController() != null) {
                parameters.getCalculationController().setProgress(i + 1, network.size());
            }
        }
        if (parameters.getCalculationController() != null) {
            parameters.getCalculationController().notifyCalculationFinished();
        }
        return influences;
    }

    /**
     * Returns the dynamic centralities of all nodes.
     *
     * @param network
     *            The network to analyze.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            Simulation parameters.
     * @return Dynamic centralities of all nodes.
     */

    /**
     * Returns the dynamic centrality of a node.
     *
     * @param network
     *            The network to analyze.
     * @param node
     *            The node to analyze.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param innersimulations
     *            The number of simulations to approximate the minimum.
     * @param parameters
     *            Simulation parameters.
     * @return Dynamic centrality of the node.
     * @throws Exception
     */
    public static double dynamicCentrality(RegulatoryNetwork network, int node, int simulations, int innersimulations,
            ConvergenceParameters parameters) throws Exception {
        return dynamicCentrality(network, node, simulations, innersimulations, parameters, network.setOfAllNodeIndices());
    }

    /**
     * Returns the dynamic centrality of a node.
     *
     * @param network
     *            The network to analyze.
     * @param node
     *            The node to analyze.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param innersimulations
     *            The number of simulations to approximate the minimum.
     * @param parameters
     *            Simulation parameters.
     * @param nodesToConsider
     *            The significant nodes.
     * @return Dynamic centrality of the node.
     * @throws Exception
     */
    public static double dynamicCentrality(RegulatoryNetwork network, int node, int simulations, int innersimulations,
            ConvergenceParameters parameters, Set<Integer> nodesToConsider) throws Exception {
        return dynamicCentrality(network, node, simulations, innersimulations, parameters, nodesToConsider, new TreeSet<Integer>());
    }

    /**
     * Returns the dynamic centrality of a node. If the node is a protected node, the total centrality is returned.
     *
     * @param network
     *            The network to analyze.
     * @param node
     *            The node to analyze.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param innersimulations
     *            The number of simulations to approximate the minimum.
     * @param parameters
     *            Simulation parameters.
     * @param nodesToConsider
     *            The significant nodes.
     * @param protectedNodes
     *            The protected nodes.
     * @return Dynamic centrality of the node.
     * @throws Exception
     */
    public static double dynamicCentrality(RegulatoryNetwork network, int node, int simulations, int innersimulations,
            ConvergenceParameters parameters, Set<Integer> nodesToConsider, Set<Integer> protectedNodes) throws Exception {
        return dynamicCentrality(network, node, simulations, innersimulations, parameters, nodesToConsider, protectedNodes,
                Double.MAX_VALUE);
    }

    /**
     * Returns the dynamic centrality of a node. If the node is a protected node, the total centrality is returned. If the calculated
     * dynamic centrality is above the threshold, the calculation is aborted.
     *
     * @param network
     *            The network to analyze.
     * @param node
     *            The node to analyze.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param innersimulations
     *            The number of simulations to approximate the minimum.
     * @param parameters
     *            Simulation parameters.
     * @param nodesToConsider
     *            The significant nodes.
     * @param protectedNodes
     *            The protected nodes.
     * @param threshold
     *            The calculation is aborted, if the dynamic centrality is above this limit.
     * @return A lower limit of the dynamic centrality of the node.
     * @throws Execution
     */
    public static double dynamicCentrality(RegulatoryNetwork network, int node, int simulations, int innersimulations,
            ConvergenceParameters parameters, Set<Integer> nodesToConsider, Set<Integer> protectedNodes, double threshold) throws Exception {
        JimenaExecutor threadPool = new JimenaExecutor(parameters.getThreads());
        ArrayList<Future<Double>> searchers = new ArrayList<Future<Double>>();

        FunctionsAnalysis enclosing = new FunctionsAnalysis();

        for (int i = 0; i < simulations; i++) {
            searchers.add(threadPool.submit(enclosing.new MinInfluence(network, node, parameters.cloneWithoutController(), nodesToConsider,
                    innersimulations, protectedNodes)));
        }

        // Wait for termination of all calculations
        threadPool.shutdown();

        while (!threadPool.isTerminated()) {
            double minResult = minResult(searchers);

            if (minResult > threshold && !Double.isInfinite(minResult)) {
                abortAll(searchers);
                try {
                    threadPool.awaitTermination(240, TimeUnit.HOURS);
                } catch (InterruptedException e1) {
                    e1.printStackTrace();
                }
                return minResult;
            }

            if (parameters.getCalculationController() != null && !parameters.getCalculationController().isOn()) {
                return Double.NaN;
            }
            setCalculationController(searchers, parameters.getCalculationController());

            try {
                Thread.sleep(500);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        try {
            threadPool.awaitTermination(240, TimeUnit.HOURS);
        } catch (InterruptedException e1) {
            e1.printStackTrace();
        }

        double sumOfDifference = 0;
        int counted = 0;
        for (int i = 0; i < simulations; i++) {
            if (!Double.isNaN(searchers.get(i).get())) {
                sumOfDifference += searchers.get(i).get();
                counted++;
            }
        }

        if (parameters.getCalculationController() != null) {
            parameters.getCalculationController().notifyCalculationFinished();
        }

        return sumOfDifference / counted;
    }

    /**
     * Returns the dynamic centrality of a connection.
     *
     * @param network
     *            The network to analyze.
     * @param connection
     *            The connection to analyze
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param innersimulations
     *            The number of simulations to approximate the minimum.
     * @param parameters
     *            Simulation parameters.
     * @return The dynamic centrality of the connection.
     * @throws Exception
     */
    public static double dynamicCentrality(RegulatoryNetwork network, NetworkConnection connection, int simulations, int innersimulations,
            ConvergenceParameters parameters) throws Exception {
        return dynamicCentrality(network, connection, simulations, innersimulations, parameters, network.setOfAllNodeIndices());
    }

    /**
     * Returns the dynamic centrality of a connection.
     *
     * @param network
     *            The network to analyze.
     * @param connection
     *            The connection to analyze
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param innersimulations
     *            The number of simulations to approximate the minimum.
     * @param parameters
     *            Simulation parameters.
     * @param nodesToConsider
     *            The significant nodes.
     * @return The dynamic centrality of the connection.
     * @throws Exception
     */
    public static double dynamicCentrality(RegulatoryNetwork network, NetworkConnection connection, int simulations, int innersimulations,
            ConvergenceParameters parameters, Set<Integer> nodesToConsider) throws Exception {
        return dynamicCentrality(network, connection, simulations, innersimulations, parameters, nodesToConsider, new TreeSet<Integer>());
    }

    /**
     * Returns the dynamic centrality of a connection. If the source node is a protected node, the total centrality is returned.
     *
     * @param network
     *            The network to analyze.
     * @param connection
     *            The connection to analyze
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param innersimulations
     *            The number of simulations to approximate the minimum.
     * @param parameters
     *            Simulation parameters.
     * @param nodesToConsider
     *            The significant nodes.
     * @param protectedNodes
     *            The protected nodes.
     * @return The dynamic centrality of the connection.
     * @throws Exception
     */
    public static double dynamicCentrality(RegulatoryNetwork network, NetworkConnection connection, int simulations, int innersimulations,
            ConvergenceParameters parameters, Set<Integer> nodesToConsider, Set<Integer> protectedNodes) throws Exception {
        return dynamicCentrality(network, connection, simulations, innersimulations, parameters, nodesToConsider, protectedNodes,
                Double.MAX_VALUE);
    }

    /**
     * Returns the dynamic centrality of a connection. If the source node is a protected node, the total centrality is returned. If the
     * calculated dynamic centrality is above the threshold, the calculation is aborted.
     *
     * @param network
     *            The network to analyze.
     * @param connection
     *            The connection to analyze
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param innersimulations
     *            The number of simulations to approximate the minimum.
     * @param parameters
     *            Simulation parameters.
     * @param nodesToConsider
     *            The significant nodes.
     * @param protectedNodes
     *            The protected nodes.
     * @param threshold
     *            The calculation is aborted, if the dynamic centrality is above this limit.
     * @return A lower limit for dynamic centrality of the connection.
     * @throws Exception
     */
    public static double dynamicCentrality(RegulatoryNetwork network, NetworkConnection connection, int simulations, int innersimulations,
            ConvergenceParameters parameters, Set<Integer> nodesToConsider, Set<Integer> protectedNodes, double threshold) throws Exception {
        JimenaExecutor threadPool = new JimenaExecutor(parameters.getThreads());
        ArrayList<Future<Double>> runs = new ArrayList<Future<Double>>();

        FunctionsAnalysis enclosing = new FunctionsAnalysis();

        for (int i = 0; i < simulations; i++) {
            runs.add(threadPool.submit(enclosing.new MinInfluence(network, connection, parameters.cloneWithoutController(),
                    nodesToConsider, innersimulations, protectedNodes)));
        }

        // Wait for termination of all calculations
        threadPool.shutdown();

        while (!threadPool.isTerminated()) {
            double minResult = minResult(runs);

            if (minResult > threshold && !Double.isInfinite(minResult)) {
                abortAll(runs);
                try {
                    threadPool.awaitTermination(240, TimeUnit.HOURS);
                } catch (InterruptedException e1) {
                    e1.printStackTrace();
                }

                return minResult;
            }

            if (parameters.getCalculationController() != null && !parameters.getCalculationController().isOn()) {
                return Double.NaN;
            }
            setCalculationController(runs, parameters.getCalculationController());

            try {
                Thread.sleep(500);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        try {
            threadPool.awaitTermination(240, TimeUnit.HOURS);
        } catch (InterruptedException e1) {
            e1.printStackTrace();
        }

        double sumOfDifference = 0;
        int counted = 0;
        for (int i = 0; i < simulations; i++) {

            if (runs.get(i).get() != null) {
                if (!Double.isNaN(runs.get(i).get())) {
                    if (!Double.isInfinite(runs.get(i).get())) {
                        sumOfDifference += runs.get(i).get();
                        counted++;
                    }
                }
            }
        }

        if (parameters.getCalculationController() != null) {
            parameters.getCalculationController().notifyCalculationFinished();
        }

        return sumOfDifference / counted;
    }

    /**
     * Checks, whether the given array of double values contains valid number.
     *
     * @param values
     *            The values to check.
     * @return True, if the values are valid.
     */
    static boolean isValid(double[] values) {
        if (values == null) {
            return false;
        }

        for (double value : values) {
            if (Double.isInfinite(value) || Double.isNaN(value)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Converts the value to a Mathematica compatible string format.
     *
     * @param value
     *            The value to convert.
     * @return A string representation-
     */
    public static String mathNumberFormat(double value) {
        return (String.valueOf(value)).replaceAll("E", "\\*^");
    }

    private static double minInfluence(RegulatoryNetwork network, int node, double[] initialVector, int simulations,
            ConvergenceParameters parameters, Set<Integer> nodesToConsider, Set<Integer> protectedNodes) throws Exception {
        RegulatoryNetwork network1 = network.cloneClean();
        RegulatoryNetwork network2 = network.cloneClean();

        for (NetworkConnection connection : network.getConnectionsBySource(node)) {
            splitConnection(network1, connection);
            splitConnection(network2, connection);
            network1.addNullMutation(connection);
        }

        double minInfluence = Double.NaN;

        for (int i = 0; i < simulations; i++) {
            double[] initVector = initialVector.clone();

            for (@SuppressWarnings("unused")
            NetworkConnection connection : network.getConnectionsBySource(node)) {
                // Do once for every connection.
                if (!protectedNodes.contains(node)) {
                    initVector = ArrayLib.add(initVector, Math.random());
                } else {
                    initVector = ArrayLib.add(initVector, initVector[node]);
                }
            }

            network1.setValues(initVector);
            network2.setValues(initVector);

            StableSteadyState n1 = new StableSteadyState(network1, parameters);
            StableSteadyState n2 = new StableSteadyState(network2, parameters);

            if (n1.call() != null && n2.call() != null) {
                double diff = MathLib.meanSquaredDifferenceUnchecked(n1.call().getResult(), n2.call().getResult(), nodesToConsider);
                minInfluence = MathLib.min(minInfluence, diff);
            }

            if ((protectedNodes.contains(node) && minInfluence != Double.MAX_VALUE) || minInfluence < 1e-15) {
                break;
            }
        }

        return minInfluence;
    }

    private static double minInfluence(RegulatoryNetwork network, NetworkConnection connection, double[] initialVector, int simulations,
            ConvergenceParameters parameters, Set<Integer> nodesToConsider, Set<Integer> protectedNodes) throws Exception {
        int source = network.getConnection(connection).getSource();

        RegulatoryNetwork network1 = network.cloneClean();
        RegulatoryNetwork network2 = network.cloneClean();

        splitConnection(network1, connection);
        splitConnection(network2, connection);

        network1.addNullMutation(connection);

        double minInfluence = Double.POSITIVE_INFINITY;

        for (int i = 0; i < simulations; i++) {
            double[] initVector = initialVector.clone();
            if (!protectedNodes.contains(source)) {
                initVector = ArrayLib.add(initVector, Math.random());
            } else {
                initVector = ArrayLib.add(initVector, initVector[source]);
            }

            network1.setValues(initVector);
            network2.setValues(initVector);

            StableSteadyState n1 = new StableSteadyState(network1, parameters);
            StableSteadyState n2 = new StableSteadyState(network2, parameters);

            if (n1.call() != null && n2.call() != null) {
                double diff = MathLib.meanSquaredDifferenceUnchecked(n1.call().getResult(), n2.call().getResult(), nodesToConsider);
                minInfluence = MathLib.min(minInfluence, diff);
            }

            if ((protectedNodes.contains(source) && minInfluence != Double.MAX_VALUE) || minInfluence < 1e-15) {
                break;
            }

        }

        return minInfluence;
    }

    /**
     * Returns the minimum dynamic centrality that can result from an array of minimum runs.
     *
     * @param runs
     *            The runs to examine.
     * @return Lower threshold for the dynamic centrality.
     * @throws Exception
     */
    private static double minResult(ArrayList<Future<Double>> runs) throws Exception {
        double sumOfDifference = 0;
        for (int i = 0; i < runs.size(); i++) {
            if (runs.get(i).isDone()) {
                sumOfDifference += runs.get(i).get();
            }
        }

        return sumOfDifference / runs.size();// counted kürzt sich weg
    }

    /**
     * Extracts the sensitivities from a influence matrix.
     *
     * @param matrix
     *            The influence matrix to analyze.
     * @return All sensitivities.
     */
    public static double[] sensitivitiesFromMatrix(double[][] matrix) {
        if (matrix == null) {
            return null;
        }

        double[] result = new double[matrix.length];
        for (int i = 0; i < matrix.length; i++) {
            result[i] = sensitivityFromMatrix(matrix, i);
        }
        return result;
    }

    /**
     * Extracts the senstivity of a given node from a influence matrix.
     *
     * @param matrix
     *            The influence matrix to analyze.
     * @param node
     *            The node whose sensitivity is to be extracted.
     * @return Sensitivity of the node.
     */
    public static double sensitivityFromMatrix(double[][] matrix, int node) {
        if (matrix == null) {
            return Double.NaN;
        }

        double result = 0;
        for (int i = 0; i < matrix.length; i++) {
            result += matrix[i][node];
        }

        return result / matrix.length;
    }

    /**
     * Update the calculation controller.
     *
     * @param searchers1
     *            Runs for the first network.
     * @param searchers2
     *            Runs for the second network.
     * @param controller
     *            The calculation controller.
     */
    static void setCalculationController(ArrayList<Future<ConvergenceResult>> searchers1, ArrayList<Future<ConvergenceResult>> searchers2,
            CalculationController controller) {
        int sum = 0;
        int done = 0;
        for (Future<ConvergenceResult> run : searchers1) {
            sum++;
            if (run.isDone()) {
                done++;
            }
        }
        for (Future<ConvergenceResult> run : searchers2) {
            sum++;
            if (run.isDone()) {
                done++;
            }
        }

        if (controller != null) {
            controller.setProgress(done, sum);
        }
    }

    /**
     * Update the calculation controller.
     *
     * @param searchers
     *            Runs for the first network.
     * @param controller
     *            The calculation controller.
     */
    private static void setCalculationController(ArrayList<Future<Double>> searchers, CalculationController controller) {
        int sum = 0;
        int done = 0;
        for (Future<Double> run : searchers) {
            sum++;
            if (run.isDone()) {
                done++;
            }
        }

        if (controller != null) {
            controller.setProgress(done, sum);
        }
    }

    /**
     * The node split function Gamma (cf. Karl et al. 2014)
     *
     * @param network
     *            The network whose connection is to be split.
     * @param connection
     *            The connection whose source node is to be split.
     */
    static void splitConnection(RegulatoryNetwork network, NetworkConnection connection) {
        int source = network.getConnection(connection).getSource();
        network.setNetworkNodes(ArrayLib.add(network.getNetworkNodes(), network.getNetworkNodes()[source].clone()));
        network.getConnection(connection).setSource(network.size() - 1);
        for (Connection i : network.getNetworkNodes()[network.size() - 1].getConnections()) {
            if (i.getSource() == source) {
                i.setSource(network.size() - 1);
            }
        }
    }

    /**
     * Returns the total centralities of all nodes.
     *
     * @param network
     *            The network to analyze.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            Simulation parameters.
     * @return Total centralities of all nodes.
     * @throws Exception
     */
    public static double[] totalCentralities(RegulatoryNetwork network, int simulations, ConvergenceParameters parameters) throws Exception {
        return centralitiesFromMatrix(totalCentralityMatrix(network, simulations, parameters));
    }

    /**
     * Mean total centrality of the network.
     *
     * @param network
     *            The network to examine.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            The simulation parameters.
     * @return The mean total centrality.
     * @throws Exception
     */
    public static double totalCentrality(RegulatoryNetwork network, int simulations, ConvergenceParameters parameters) throws Exception {
        double sum = 0;
        for (int i = 0; i < network.size(); i++) {
            sum += totalCentrality(network, i, simulations, parameters);
            if (Double.isNaN(sum)) {
                break;
            }
            if (parameters.getCalculationController() != null && !parameters.getCalculationController().isOn()) {
                return Double.NaN;
            }
            if (parameters.getCalculationController() != null) {
                parameters.getCalculationController().setProgress(i + 1, network.size());
            }

        }
        if (parameters.getCalculationController() != null) {
            parameters.getCalculationController().notifyCalculationFinished();
        }
        return sum / network.size();
    }

    /**
     * Total centrality of a network node.
     *
     * @param network
     *            The network to examine.
     * @param node
     *            The network node whose centrality is to be calculated.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            The simulation parameters.
     * @return The total centrality of the node.
     * @throws Exception
     */
    public static double totalCentrality(RegulatoryNetwork network, int node, int simulations, ConvergenceParameters parameters)
            throws Exception {
        return totalCentrality(network, node, simulations, parameters, network.setOfAllNodeIndices());
    }

    /**
     * Total centrality of a network node.
     *
     * @param network
     *            The network to examine.
     * @param node
     *            The network node whose centrality is to be calculated.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            The simulation parameters.
     * @param nodesToConsider
     *            The significant nodes.
     * @return The total centrality of the node.
     * @throws Exception
     */
    public static double totalCentrality(RegulatoryNetwork network, int node, int simulations, ConvergenceParameters parameters,
            Set<Integer> nodesToConsider) throws Exception {
        ArrayList<NetworkConnection> connections = network.getConnectionsBySource(node);

        RegulatoryNetwork mutated = network.cloneClean();
        mutated.addNullMutation(connections);

        return ConvergenceComparator.difference(network, mutated, simulations, parameters);
    }

    /**
     * Total centrality of a connection.
     *
     * @param network
     *            The network to examine.
     * @param connection
     *            The connection whose centrality is to be calculated.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            The simulation parameters.
     * @return Total centrality of the connection.
     * @throws Exception
     */
    public static double totalCentrality(RegulatoryNetwork network, NetworkConnection connection, int simulations,
            ConvergenceParameters parameters) throws Exception {
        return totalCentrality(network, connection, simulations, parameters, network.setOfAllNodeIndices());
    }

    /**
     * Total centrality of a connection.
     *
     * @param network
     *            The network to examine.
     * @param connection
     *            The connection whose centrality is to be calculated.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            The simulation parameters.
     * @param nodesToConsider
     *            The significant nodes.
     * @return Total centrality of the connection.
     * @throws Exception
     */
    public static double totalCentrality(RegulatoryNetwork network, NetworkConnection connection, int simulations,
            ConvergenceParameters parameters, Set<Integer> nodesToConsider) throws Exception {
        RegulatoryNetwork mutated = network.cloneClean();
        mutated.addNullMutation(connection);
        return ConvergenceComparator.difference(network, mutated, simulations, parameters);
    }

    /**
     * Returns a total centrality influence matrix.
     *
     * @param network
     *            The network to examine.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            The simulation parameters.
     * @return Total centrality influence matrix.
     * @throws Exception
     */
    public static double[][] totalCentralityMatrix(RegulatoryNetwork network, int simulations, ConvergenceParameters parameters)
            throws Exception {

        double[][] matrix = new double[network.size()][];
        for (int i = 0; i < network.size(); i++) {
            matrix[i] = totalCentralityVector(network, i, simulations, parameters.cloneWithoutController());
            while (!isValid(matrix[i])) {
                matrix[i] = totalCentralityVector(network, i, simulations, parameters.cloneWithoutController());
            }
            if (parameters.getCalculationController() != null && !parameters.getCalculationController().isOn()) {
                return null;
            }
            if (parameters.getCalculationController() != null) {
                parameters.getCalculationController().setProgress(i + 1, network.size());
            }
        }
        if (parameters.getCalculationController() != null) {
            parameters.getCalculationController().notifyCalculationFinished();
        }
        return matrix;
    }

    /**
     * Total centrality of a network node.
     *
     * @param network
     *            The network to examine.
     * @param node
     *            The node whose centrality is to be calculated.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            The simulation parameters.
     * @return Total centrality of the node
     * @throws Exception
     */
    public static double[] totalCentralityVector(RegulatoryNetwork network, int node, int simulations, ConvergenceParameters parameters)
            throws Exception {
        ArrayList<NetworkConnection> connections = network.getConnectionsBySource(node);

        RegulatoryNetwork mutated = network.cloneClean();
        mutated.addNullMutation(connections);

        return ConvergenceComparator.differenceV(network, mutated, simulations, parameters);
    }

    /**
     * Returns the value centralities of all nodes.
     *
     * @param network
     *            The network to analyze.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            Simulation parameters.
     * @return Value centralities of all nodes.
     * @throws Exception
     */
    public static double[] valueCentralities(RegulatoryNetwork network, int simulations, ConvergenceParameters parameters) throws Exception {
        return centralitiesFromMatrix(valueCentralityMatrix(network, simulations, parameters));
    }

    /**
     * Returns the total value centrality of the network.
     *
     * @param network
     *            The network to examine.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            The simulation parameters.
     * @return Total value centrality of the network.
     * @throws Exception
     */
    public static double valueCentrality(RegulatoryNetwork network, int simulations, ConvergenceParameters parameters) throws Exception {
        double sum = 0;
        for (int i = 0; i < network.size(); i++) {

            sum += valueCentrality(network, i, simulations, parameters.cloneWithoutController());
            if (parameters.getCalculationController() != null && !parameters.getCalculationController().isOn()) {
                return Double.NaN;
            }
            if (parameters.getCalculationController() != null) {
                parameters.getCalculationController().setProgress(i + 1, network.size());
            }
        }

        if (parameters.getCalculationController() != null) {
            parameters.getCalculationController().notifyCalculationFinished();
        }

        return sum / network.size();
    }

    /**
     * Returns the value centrality of a node.
     *
     * @param network
     *            The network to examine.
     * @param node
     *            The network node whose centrality is to be calculated.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            The simulation parameters.
     * @return Value centrality of the node.
     * @throws Exception
     */
    public static double valueCentrality(RegulatoryNetwork network, int node, int simulations, ConvergenceParameters parameters)
            throws Exception {
        return valueCentrality(network, node, simulations, parameters, network.setOfAllNodeIndices());
    }

    /**
     * Returns the value centrality of a node.
     *
     * @param network
     *            The network to examine.
     * @param node
     *            The network node whose centrality is to be calculated.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            The simulation parameters.
     * @param nodesToConsider
     *            The significant nodes.
     * @return Value centrality of the node.
     * @throws Exception
     */
    public static double valueCentrality(RegulatoryNetwork network, int node, int simulations, ConvergenceParameters parameters,
            Set<Integer> nodesToConsider) throws Exception {
        double[] temp = valueCentralityVector(network, node, simulations, parameters);
        double result = 0;
        int counter = 0;
        for (Integer i : nodesToConsider) {
            result += temp[i];
            counter++;
        }
        return result / counter;
    }

    /**
     * Returns the value centrality of a connection.
     *
     * @param network
     *            The network to examine.
     * @param connection
     *            The connection to examine.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            The simulation parameters.
     * @return Value centrality of the connection.
     * @throws Exception
     */
    public static double valueCentrality(RegulatoryNetwork network, NetworkConnection connection, int simulations,
            ConvergenceParameters parameters) throws Exception {
        return MathLib.average(valueCentralityVector(network, connection, simulations, parameters));
    }

    /**
     * Returns the value centrality of a connection.
     *
     * @param network
     *            The network to examine.
     * @param connection
     *            The connection to examine.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            The simulation parameters.
     * @param nodesToConsider
     *            The significant nodes.
     * @return Value centrality of the connection.
     * @throws Exception
     */
    public static double valueCentrality(RegulatoryNetwork network, NetworkConnection connection, int simulations,
            ConvergenceParameters parameters, Set<Integer> nodesToConsider) throws Exception {
        return MathLib.average(valueCentralityVector(network, connection, simulations, parameters), nodesToConsider);
    }

    /**
     * Returns a value centrality influence matrix of the network.
     *
     * @param network
     *            The network to examine.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            The simulation parameters.
     * @return Value centrality influence matrix.
     * @throws Exception
     */
    public static double[][] valueCentralityMatrix(RegulatoryNetwork network, int simulations, ConvergenceParameters parameters)
            throws Exception {
        double[][] matrix = new double[network.size()][];
        for (int i = 0; i < network.size(); i++) {
            matrix[i] = valueCentralityVector(network, i, simulations, parameters.cloneWithoutController());
            while (!isValid(matrix[i])) {
                matrix[i] = valueCentralityVector(network, i, simulations, parameters.cloneWithoutController());
            }
            if (parameters.getCalculationController() != null && !parameters.getCalculationController().isOn()) {
                return null;
            }
            if (parameters.getCalculationController() != null) {
                parameters.getCalculationController().setProgress(i + 1, network.size());
            }
        }
        if (parameters.getCalculationController() != null) {
            parameters.getCalculationController().notifyCalculationFinished();
        }
        return matrix;
    }

    /**
     * The value centrality of a given node on all nodes.
     *
     * @param network
     *            The network to examine.
     * @param node
     *            The network node whose centrality is to be calculated.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            The simulation parameters.
     * @return A vector of the centrality on all nodes.
     * @throws Exception
     */
    public static double[] valueCentralityVector(RegulatoryNetwork network, int node, int simulations, ConvergenceParameters parameters)
            throws Exception {
        JimenaExecutor threadPool = new JimenaExecutor(parameters.getThreads());
        ArrayList<Future<ConvergenceResult>> searchers1 = new ArrayList<Future<ConvergenceResult>>();
        ArrayList<Future<ConvergenceResult>> searchers2 = new ArrayList<Future<ConvergenceResult>>();

        for (int i = 0; i < simulations; i++) {
            double[] startValues1 = MathLib.randomVector(network.size());
            double[] startValues2 = startValues1.clone();
            startValues2[node] = Math.random();

            RegulatoryNetwork networkN1 = network.cloneClean();
            RegulatoryNetwork networkN2 = network.cloneClean();

            networkN1.setValues(startValues1);
            networkN2.setValues(startValues2);

            // Not using the steadyStableState method avoids copying the networks
            searchers1.add(threadPool.submit(new StableSteadyState(networkN1, parameters.cloneWithoutController())));
            searchers2.add(threadPool.submit(new StableSteadyState(networkN2, parameters.cloneWithoutController())));
        }

        // Wait for termination of all calculations
        threadPool.shutdown();

        while (!threadPool.isTerminated()) {
            try {
                Thread.sleep(500);
                if (parameters.getCalculationController() != null && !parameters.getCalculationController().isOn()) {
                    return null;
                }
                setCalculationController(searchers1, searchers2, parameters.getCalculationController());

            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        double[] sumOfDifference = new double[network.size()];
        int counted = 0;
        for (int i = 0; i < simulations; i++) {
            if (searchers1.get(i).get() != null && searchers2.get(i).get() != null) {
                MathLib.addDoubleArraysUnchecked(sumOfDifference,
                        MathLib.squaredDifferenceUnchecked(searchers1.get(i).get().getResult(), searchers2.get(i).get().getResult()));
                counted++;
            }
        }

        MathLib.scaleUnchecked(sumOfDifference, 1d / counted);

        if (parameters.getCalculationController() != null) {
            parameters.getCalculationController().notifyCalculationFinished();
        }

        return sumOfDifference;
    }

    /**
     * The value centrality of a given connection on all nodes.
     *
     * @param network
     *            The network to examine.
     * @param connection
     *            The network connection whose centrality is to be calculated.
     * @param simulations
     *            The number of simulations to approximate the integral.
     * @param parameters
     *            The simulation parameters.
     * @return A vector of the centrality on all nodes.
     * @throws Exception
     */
    public static double[] valueCentralityVector(RegulatoryNetwork network, NetworkConnection connection, int simulations,
            ConvergenceParameters parameters) throws Exception {
        int source = network.getConnection(connection).getSource();
        JimenaExecutor threadPool = new JimenaExecutor(parameters.getThreads());
        ArrayList<Future<ConvergenceResult>> searchers1 = new ArrayList<Future<ConvergenceResult>>();
        ArrayList<Future<ConvergenceResult>> searchers2 = new ArrayList<Future<ConvergenceResult>>();

        for (int i = 0; i < simulations; i++) {
            double[] startValues1 = MathLib.randomVector(network.size());
            double[] startValues2 = startValues1.clone();

            startValues1 = ArrayLib.add(startValues1, startValues1[source]);
            startValues2 = ArrayLib.add(startValues2, Math.random());

            RegulatoryNetwork networkN1 = network.cloneClean();
            RegulatoryNetwork networkN2 = network.cloneClean();

            splitConnection(networkN1, connection);
            splitConnection(networkN2, connection);

            networkN1.setValues(startValues1);
            networkN2.setValues(startValues2);

            // Not using the steadyStableState method avoids copying the networks
            searchers1.add(threadPool.submit(new StableSteadyState(networkN1, parameters.cloneWithoutController())));
            searchers2.add(threadPool.submit(new StableSteadyState(networkN2, parameters.cloneWithoutController())));
        }

        // Wait for termination of all calculations
        threadPool.shutdown();

        while (!threadPool.isTerminated()) {
            try {
                Thread.sleep(500);
                if (parameters.getCalculationController() != null && !parameters.getCalculationController().isOn()) {
                    return null;
                }
                setCalculationController(searchers1, searchers2, parameters.getCalculationController());

            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        double[] sumOfDifference = new double[network.size()];
        int counted = 0;
        for (int i = 0; i < simulations; i++) {
            if (searchers1.get(i).get() != null && searchers2.get(i).get() != null) {
                MathLib.addDoubleArraysUnchecked(sumOfDifference,
                        MathLib.squaredDifferenceUnchecked(searchers1.get(i).get().getResult(), searchers2.get(i).get().getResult()));
                counted++;
            }
        }
        MathLib.scaleUnchecked(sumOfDifference, 1d / counted);

        if (parameters.getCalculationController() != null) {
            parameters.getCalculationController().notifyCalculationFinished();
        }

        return sumOfDifference;
    }
}
