package jimena.analysis;

import java.awt.Rectangle;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Random;

import jimena.binarybf.BinaryBooleanFunction;
import jimena.binarybf.actinhibitf.ActivatorInhibitorFunction;
import jimena.binaryrn.Connection;
import jimena.binaryrn.NetworkNode;
import jimena.binaryrn.RegulatoryNetwork;
import jimena.libs.DoubleValue;

/**
 * A class generating random Erdös-Renyi and scale-free network for the Jimena framework.
 *
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 *
 */
public class NetworkGenerator {
    /**
     * Contructs an Erd�s�R�nyi network with given properties.
     *
     * @param size
     *            Size of the network.
     * @param connectionsWithoutInputLoops
     *            Number of connections excluding input loops
     * @param nonInputLoops
     *            Number of loops excluding input loops
     * @param addInputLoops
     *            whether to add input loops or not.
     * @param act
     *            Share of activating influences.
     * @param inputs
     *            Number of input nodes.
     * @return An random ER network.
     */
    public static RegulatoryNetwork constructERnetwork(int size, int connectionsWithoutInputLoops, int nonInputLoops,
            boolean addInputLoops, double act, int inputs) {
        RegulatoryNetwork n = null;
        do {
            n = constructERnetwork(size, connectionsWithoutInputLoops, nonInputLoops, addInputLoops, act);
        } while (n.numberOfInputs() != inputs);
        return n;
    }

    /**
     * Contructs an Erd�s�R�nyi network with given properties.
     *
     * @param size
     *            Size of the network.
     * @param connectionsWithoutInputLoops
     *            Number of connections excluding input loops
     * @param nonInputLoops
     *            Number of loops excluding input loops
     * @param addInputLoops
     *            whether to add input loops or not.
     * @param act
     *            Share of activating influences.
     * @return An random ER network.
     */
    public static RegulatoryNetwork constructERnetwork(int size, int connectionsWithoutInputLoops, int nonInputLoops,
            boolean addInputLoops, double act) {
        Random generator = new Random();
        ArrayList<Integer> from;
        ArrayList<Integer> to;
        ArrayList<Boolean> activating;

        int failsafe = 100000;
        do {
            failsafe--;
            if (failsafe == 0) {
                return null;
            }

            from = new ArrayList<Integer>();
            to = new ArrayList<Integer>();
            activating = new ArrayList<Boolean>();
            for (int i = 0; i < connectionsWithoutInputLoops; i++) {
                int n1 = generator.nextInt(size);
                int n2 = generator.nextInt(size);
                if (n1 == n2 || (contains(from, to, n1, n2))) {
                    i--;
                    continue;
                }
                from.add(n1);
                to.add(n2);
                activating.add(Math.random() < act);
            }
        } while (!isConnected(from, to, size));

        for (int i = 0; i < nonInputLoops; i++) {
            failsafe--;
            if (failsafe == 0) {
                return null;
            }
            int n = generator.nextInt(size);
            if (contains(from, to, n, n)) {
                i--;
                continue;
            }
            from.add(n);
            to.add(n);
            activating.add(true);
        }

        if (addInputLoops) {
            addInputLoops(from, to, activating, size);
        }

        return constructNetwork(from, to, activating, size);
    }

    /**
     * Adds a loops to all input nodes that do not yet have one.
     *
     * @param from
     *            Array indicating the source nodes of the connections.
     * @param to
     *            Array indicating the target nodes of the connections.
     * @param activating
     *            Array indicating whether the connection is inhibiting.
     * @param size
     *            Size of the network.
     */
    public static void addInputLoops(ArrayList<Integer> from, ArrayList<Integer> to, ArrayList<Boolean> activating, int size) {
        for (int i = 0; i < size; i++) {
            if (isInput(to, i)) {
                from.add(i);
                to.add(i);
                activating.add(true);
            }
        }
    }

    /**
     * Contructs a scale-free network with given properties.
     *
     * @param size
     *            Size of the network.
     * @param connectionsWithoutInputLoops
     *            Number of connections excluding input loops
     * @param nonInputLoops
     *            Number of loops excluding input loops
     * @param addInputLoops
     *            whether to add input loops or not.
     * @param act
     *            Share of activating influences.
     * @param inputs
     *            Number of input nodes.
     * @return An random ER network.
     */
    public static RegulatoryNetwork constructSFnetwork(int size, int connectionsWithoutInputLoops, int nonInputLoops,
            boolean addInputLoops, double act, int inputs) {
        RegulatoryNetwork n = null;
        do {
            n = constructSFnetwork(size, connectionsWithoutInputLoops, nonInputLoops, addInputLoops, act);
        } while (n.numberOfInputs() != inputs);
        return n;
    }

    /**
     * Contructs a scale-free network with given properties.
     *
     * @param size
     *            Size of the network.
     * @param connectionsWithoutInputLoops
     *            Number of connections excluding input loops
     * @param nonInputLoops
     *            Number of loops excluding input loops
     * @param addInputLoops
     *            whether to add input loops or not.
     * @param act
     *            Share of activating influences.
     * @return An random ER network.
     */
    public static RegulatoryNetwork constructSFnetwork(int size, int connectionsWithoutAnyLoops, int nonInputLoops, boolean addInputLoops,
            double act) {
        int perStep = (int) Math.ceil((((double) (connectionsWithoutAnyLoops - 1)) / size));

        Random generator = new Random();
        ArrayList<Integer> from = new ArrayList<Integer>();
        ArrayList<Integer> to = new ArrayList<Integer>();
        ArrayList<Boolean> activating = new ArrayList<Boolean>();

        from.add(0);
        to.add(1);
        activating.add(Math.random() < act);

        int deferred = 0;
        int failsafe = 100000;

        for (int i = 0; i < size; i++) {
            int tries = 0;
            for (int c = 0; c < (perStep + deferred); c++) {
                failsafe--;
                if (failsafe == 0) {
                    return null;
                }

                int otherNode;
                if (Math.random() < 0.5) {
                    otherNode = from.get(generator.nextInt(from.size()));
                } else {
                    otherNode = to.get(generator.nextInt(from.size()));
                }

                boolean incoming = Math.random() < 0.5;

                int f, t;

                if (incoming) {
                    f = otherNode;
                    t = i;
                } else {
                    f = i;
                    t = otherNode;
                }

                if (!contains(from, to, f, t) && f != t) {
                    from.add(f);
                    to.add(t);
                    activating.add(Math.random() < act);
                } else {
                    c--;
                }
                tries++;
                if (tries >= 1000) {
                    deferred = perStep + deferred - (c + 1);
                    break;
                }

                if ((c + 1) == (perStep + deferred)) {
                    deferred = 0;
                }
            }
        }

        while (from.size() > connectionsWithoutAnyLoops) {
            failsafe--;
            if (failsafe == 0) {
                return null;
            }

            // Deleting with equal probability preserves the distribution
            int deleted = generator.nextInt(from.size());
            if (noDisconnections(from, to, from.get(deleted), to.get(deleted), size)) {
                from.remove(deleted);
                to.remove(deleted);
                activating.remove(deleted);
            }
        }

        for (int i = 0; i < nonInputLoops; i++) {
            failsafe--;
            if (failsafe == 0) {
                return null;
            }

            int otherNode;
            if (Math.random() < 0.5) {
                otherNode = from.get(generator.nextInt(from.size()));
            } else {
                otherNode = to.get(generator.nextInt(from.size()));
            }
            if (!contains(from, to, otherNode, otherNode) && !isInput(to, otherNode)) {
                from.add(otherNode);
                to.add(otherNode);
                activating.add(true);
            } else {
                i--;
            }
        }

        if (addInputLoops) {
            for (int i = 0; i < size; i++) {
                if (isInput(to, i)) {
                    from.add(i);
                    to.add(i);
                    activating.add(true);
                }
            }
        }

        return constructNetwork(from, to, activating, size);
    }

    /**
     * Creates a network from the given lists:
     *
     * @param from
     *            Array indicating the source nodes of the connections.
     * @param to
     *            Array indicating the target nodes of the connections.
     * @param activating
     *            Array indicating whether the connection is inhibiting.
     * @param size
     *            Size of the network.
     * @return A new network with the given specification.
     */
    public static RegulatoryNetwork constructNetwork(ArrayList<Integer> from, ArrayList<Integer> to, ArrayList<Boolean> activating, int size) {
        NetworkNode[] nodes = new NetworkNode[size];

        DoubleValue timeIndex = new DoubleValue(0);

        for (int i = 0; i < size; i++) {
            ArrayList<Integer> connections = new ArrayList<Integer>();
            ArrayList<Boolean> act = new ArrayList<Boolean>();
            for (int j = 0; j < to.size(); j++) {
                if (to.get(j) == i) {
                    connections.add(from.get(j));
                    act.add(activating.get(j));
                }
            }
            boolean[] actarray = new boolean[act.size()];
            for (int j = 0; j < act.size(); j++) {
                actarray[j] = act.get(j);
            }

            Connection[] connection = new Connection[act.size()];
            for (int j = 0; j < act.size(); j++) {
                connection[j] = new Connection(connections.get(j), new Point2D.Double[0]);
            }

            BinaryBooleanFunction function = new ActivatorInhibitorFunction(actarray);
            nodes[i] = new NetworkNode("n" + String.valueOf(i), function, connection, new Rectangle.Double(), 0, timeIndex);
        }

        return new RegulatoryNetwork(nodes, timeIndex);
    }

    /**
     * Checks whether a connections is contained a list of connections.
     *
     * @param from
     *            Array indicating the source nodes of the connections.
     * @param to
     *            Array indicating the target nodes of the connections.
     * @param f
     *            Source node of the query.
     * @param t
     *            Target node of the query
     * @return True, if the connections is contained in the list.
     */
    public static boolean contains(ArrayList<Integer> from, ArrayList<Integer> to, int f, int t) {
        for (int i = 0; i < from.size(); i++) {
            if (from.get(i) == f && to.get(i) == t) {
                return true;
            }
        }
        return false;
    }

    /**
     * Checks, whether a given network specification corresponds to a fully connected network.
     *
     * @param from
     *            Array indicating the source nodes of the connections.
     * @param to
     *            Array indicating the target nodes of the connections.
     * @param size
     *            Size of the network
     * @return True, if the network is connected.
     */
    public static boolean isConnected(ArrayList<Integer> from, ArrayList<Integer> to, int size) {
        boolean[] connected = new boolean[size];
        connected[0] = true;
        boolean change = true;
        while (change) {
            change = false;
            for (int i = 0; i < from.size(); i++) {
                if (connected[from.get(i)]) {
                    if (connected[to.get(i)] == false) {
                        connected[to.get(i)] = true;
                        change = true;
                    }
                }
                if (connected[to.get(i)]) {
                    if (connected[from.get(i)] == false) {
                        connected[from.get(i)] = true;
                        change = true;
                    }
                }
            }
        }

        for (boolean c : connected) {
            if (!c) {
                return false;
            }
        }
        return true;
    }

    /**
     * Checks, whether a given network specification corresponds to a fully connected network if a connections is removed.
     *
     * @param from
     *            Array indicating the source nodes of the connections.
     * @param to
     *            Array indicating the target nodes of the connections.
     * @param size
     *            Size of the network
     * @param f
     *            Source node of the connection to be removed.
     * @param t
     *            Target node of the connection to be removed.
     * @return True, if the network is connected.
     */
    @SuppressWarnings("unchecked")
    public static boolean noDisconnections(ArrayList<Integer> from, ArrayList<Integer> to, int f, int t, int size) {
        ArrayList<Integer> fromCopy = (ArrayList<Integer>) from.clone();
        ArrayList<Integer> toCopy = (ArrayList<Integer>) to.clone();

        for (int i = 0; i < from.size(); i++) {
            if (from.get(i) == f && to.get(i) == t) {
                fromCopy.remove(i);
                toCopy.remove(i);
            }
        }

        return isConnected(fromCopy, toCopy, size);
    }

    /**
     * Checks, whether a given node is a input to network specification.
     *
     * @param to
     *            Array indicating the target nodes of the connections.
     * @param node
     *            The node to check
     * @return True, if the node is a input.
     */
    public static boolean isInput(ArrayList<Integer> to, int node) {
        for (int i : to) {
            if (i == node) {
                return false;
            }
        }
        return true;
    }
}
