package jimena.binarybf.actinhibitf;

import jimena.binarybf.BinaryBooleanFunction;
import jimena.binarybf.treebf.TreeBooleanFunction;
import net.sf.javabdd.BDD;
import net.sf.javabdd.BDDFactory;

/**
 * A immutable interpolatable boolean function represented internally as a set of activator and a set of inhibitors. The function is true
 * iff at least one activator is true and no inhibitors are true.
 *
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 */
public class ActivatorInhibitorFunction extends BinaryBooleanFunction {
    private static final long serialVersionUID = -1756565073860160320L;
    private boolean[] activators;
    private boolean[] disabled;
    private TreeBooleanFunction treeBooleanFunction = null;

    /**
     * Creates a ActivatorInhibitorFunction from a list of activators and inhibitors. If an empty array is provided the constant false
     * function is created.
     *
     * @param activators
     *            List of activators and inhibitors
     * @param disabled
     *            The list of disabled connectionss
     */
    public ActivatorInhibitorFunction(boolean[] activators, boolean[] disabled) {
        this(activators);
        if (disabled.length != activators.length) {
            throw new IllegalArgumentException("The length of the activators arrays must correspond to the length of the disabled array");
        }
        this.disabled = disabled;
        treeBooleanFunction = new TreeBooleanFunction(activators);
        for (int i = 0; i < disabled.length; i++) {
            if (disabled[i]) {
                treeBooleanFunction.disableConnection(i);
            }
        }
    }

    /**
     * Creates a ActivatorInhibitorFunction from a list of activators and inhibitors. If an empty array is provided the constant false
     * function is created.
     *
     * @param activators
     *            List of activators and inhibitors
     */
    public ActivatorInhibitorFunction(boolean[] activators) {
        if (activators == null) {
            throw new NullPointerException("Please provide an empty array to create the nullary constant false function.");
        }

        this.activators = activators;
        this.disabled = new boolean[activators.length];
        treeBooleanFunction = new TreeBooleanFunction(activators);
    }

    @Override
    public boolean eval(boolean[] values) {
        boolean result = false;

        for (int i = 0; i < activators.length; i++) {
            if (!disabled[i]) {
                if (activators[i] == true && values[i] == true) {
                    result = true;
                    break;
                }
            }
        }

        if (result == false) {
            return false;
        }

        for (int i = 0; i < activators.length; i++) {
            if (!disabled[i]) {
                if (activators[i] == false && values[i] == true) {
                    return false;
                }
            }
        }

        return result;
    }

    @Override
    public int getArity() {
        return activators.length;
    }

    private boolean onlyInhibitors(boolean[] activators, boolean[] disabled) {
        for (int i = 0; i < activators.length; i++) {
            if (activators[i] && !disabled[i]) {
                return false;
            }
        }
        return true;
    }

    private boolean onlyActivators(boolean[] activators, boolean[] disabled) {
        for (int i = 0; i < activators.length; i++) {
            if (!activators[i] && !disabled[i]) {
                return false;
            }
        }
        return true;
    }

    private boolean allDeactivated(boolean[] disabled) {
        for (boolean b : disabled) {
            if (!b) {
                return false;
            }
        }
        return true;
    }

    @Override
    public double interpolateSQUAD(double[] input, double[] squadWeights) {
        // The nullary constant false function.
        if (allDeactivated(disabled)) {
            return 0;
        }

        // Ignored the typos in the odefy and the SQUAD paper, used http://www.ncbi.nlm.nih.gov/pmc/articles/PMC1440308/?tool=pubmed

        double result = 0;

        if (onlyActivators(activators, disabled)) {
            double actsum = 0;
            double actweight = 0;

            for (int i = 0; i < activators.length; i++) {
                if (disabled[i]) {
                    continue;
                }

                try {
                    actsum += squadWeights[i];
                    actweight += squadWeights[i] * input[i];
                } catch (ArrayIndexOutOfBoundsException e) {
                    // sum +=0
                }
            }

            // Only activators
            result = ((1 + actsum) / actsum) * (actweight / (1 + actweight));
        } else if (onlyInhibitors(activators, disabled)) {
            double inhibitsum = 0;
            double inhibitweight = 0;

            for (int i = 0; i < activators.length; i++) {
                if (disabled[i]) {
                    continue;
                }

                try {
                    inhibitsum += squadWeights[i];
                    inhibitweight += squadWeights[i] * input[i];
                } catch (ArrayIndexOutOfBoundsException e) {
                    // sum +=0
                }
            }

            result = 1 - ((1 + inhibitsum) / (inhibitsum)) * (inhibitweight / (1 + inhibitweight));
        } else {
            double actsum = 0;
            double actweight = 0;
            double inhibitsum = 0;
            double inhibitweight = 0;

            for (int i = 0; i < activators.length; i++) {
                if (disabled[i]) {
                    continue;
                }
                try {
                    if (activators[i]) {
                        actsum += squadWeights[i];
                        actweight += squadWeights[i] * input[i];
                    } else {
                        inhibitsum += squadWeights[i];
                        inhibitweight += squadWeights[i] * input[i];
                    }
                } catch (ArrayIndexOutOfBoundsException e) {
                    // sum +=0
                }
            }

            result = ((1 + actsum) / actsum) * (actweight / (1 + actweight));
            result *= 1 - ((1 + inhibitsum) / (inhibitsum)) * (inhibitweight / (1 + inhibitweight));
        }

        return result;
    }

    @Override
    public double interpolateBooleCube(double[] input) {
        return getTreeBooleanFunction().interpolateBooleCube(input);
    }

    /**
     * Returns a tree boolean function for this function.
     *
     * @return The tree boolean function.
     */
    private TreeBooleanFunction getTreeBooleanFunction() {
        return treeBooleanFunction;
    }

    @Override
    public ActivatorInhibitorFunction clone() {
        // Faster than copying the tree.
        return new ActivatorInhibitorFunction(activators.clone(), disabled.clone());
    }

    @Override
    protected BDD createBDDUnchecked(BDD[] inputs, BDDFactory bddFactory) {
        return getTreeBooleanFunction().createBDD(inputs, bddFactory);
    }

    @Override
    public String toString() {
        return getTreeBooleanFunction().toString();
    }

    private boolean wasMutated = false;

    @Override
    public void disableConnection(int position) {
        disabled[position] = true;
        treeBooleanFunction.disableConnection(position);
        wasMutated = true;
    }

    @Override
    public void restore() {
        if (!wasMutated) {
            return;
        }
        disabled = new boolean[disabled.length];
        treeBooleanFunction.restore();
    }

    @Override
    public String getFunctionString(String[] nodeNames, boolean odefyCompatible) {
        return getTreeBooleanFunction().getFunctionString(nodeNames, odefyCompatible);
    }

    /**
     * Returns the array of activators of the function.
     *
     * @return Array of activators of the function.
     */
    public boolean[] getActivators() {
        return activators;
    }

}
