package jimena.binarybf.treebf;

import java.util.ArrayList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jimena.binarybf.BinaryBooleanFunction;
import jimena.libs.PatternLib;
import jimena.libs.SearchResult;
import net.sf.javabdd.BDD;
import net.sf.javabdd.BDDFactory;

/**
 * A interpolatable boolean function represented internally as a standard binary tree of logic gates of the type Node.
 * 
 * Suitable for fast interpolations, especially of nodes with many inputs.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class TreeBooleanFunction extends BinaryBooleanFunction {
    private static final long serialVersionUID = 7224250583219201061L;

    private TreeNode root;
    private TreeNode originalTree;
    private int arity;

    /**
     * Returns the arity of the boolean function.
     * 
     * @return Arity of the boolean
     */
    @Override
    public int getArity() {
        return arity;
    }

    /**
     * Backs up the tree so it can be restored after mutations.
     * 
     */
    private void backupTree() {
        originalTree = root.clone();
    }

    /**
     * Creates a boolean function from a boolean expression.
     * 
     * In the boolean expression the inputs to the function are noted as non-negative integers (0, 1, 2,...), the AND, OR and NOT gates are
     * noted as &&, || and ! respectively. As usual, round brackets () can be used to group expressions and spaces are ignored. The literals
     * "true" and "false" may be used.
     * 
     * Examples:
     * 
     * f(a,b,c) = ¬(a^b)vc => "!(0 && 1) || 2"
     * 
     * @param expression
     *            A boolean expression
     * @param arity
     *            Arity of the created function
     */
    public TreeBooleanFunction(String expression, int arity) {
        // Checks done by the parse function
        root = parse(expression);
        if (root.getMinArity() > arity) {
            throw new IllegalArgumentException("Arity of the function must be greater than the index of the highest input.");
        }
        this.arity = arity;
        backupTree();
    }

    /**
     * Cf. {@link TreeBooleanFunction#TreeBooleanFunction(String, int)}.
     * 
     * @param expression
     *            A boolean expression
     */
    public TreeBooleanFunction(String expression) {
        // Checks done by the parse function
        root = parse(expression);
        this.arity = root.getMinArity();
        backupTree();
    }

    /**
     * Cf. {@link TreeBooleanFunction#TreeBooleanFunction(TreeNode, int)}.
     * 
     * @param root
     *            The root of the tree structure
     */
    public TreeBooleanFunction(TreeNode root) {
        // Checks done by the other constructor
        this(root, root.getMinArity());
    }

    /**
     * Creates a boolean function from a tree structure. Will throw a IllegalArgumentException if there are feedback loops in the tree
     * structure (Which should not be constructible anyway).
     * 
     * @param root
     *            The root of the tree structure
     * @param arity
     *            Arity of the created function
     */
    public TreeBooleanFunction(TreeNode root, int arity) {
        if (root == null) {
            throw new NullPointerException();
        }
        if (root.isCyclic()) {
            throw new IllegalArgumentException("The tree of a boolean function must be a acyclic.");
        }

        this.root = root;
        this.arity = arity;
        backupTree();
    }

    /**
     * Creates a TreeBooleanFunction from a list of activators and inhibitors.
     * 
     * @param activators
     *            List of activators and inhibitors.
     */
    public TreeBooleanFunction(boolean[] activators) {
        if (activators == null) {
            throw new NullPointerException("Provide an empty array instead of null to create the nullary constant-false function");
        }

        ArrayList<TreeNode> activatorNodes = new ArrayList<TreeNode>();
        ArrayList<TreeNode> inhibitorNodes = new ArrayList<TreeNode>();

        for (int i = 0; i < activators.length; i++) {
            if (activators[i]) {
                activatorNodes.add(new InputNode(i));
            } else {
                inhibitorNodes.add(new InputNode(i));
            }
        }

        if (activatorNodes.size() == 0 && inhibitorNodes.size() == 0) {
            // if there are no inputs, assume a constant false node
            root = new ConstantNode(false);
            this.arity = 0;
        } else {
            root = TreeNodeLib.getActInhibitTree(activatorNodes, inhibitorNodes);
            this.arity = activators.length;
        }

        // Not slower than directly constructing the tree here.
        backupTree();
    }

    /**
     * Parses a boolean expression which satisfies the format described at {@link TreeBooleanFunction#TreeBooleanFunction(String)} .
     * 
     * @param expression
     *            A boolean expression
     * @return The root of a binary tree of logic gate representing the function
     */
    private TreeNode parse(String expression) {
        // Remove enclosing brackets and spaces
        expression = expression.trim();

        // The recursive call removes multiple brackets
        if (expression.startsWith("(") && expression.endsWith(")")) {
            return parse(expression.substring(1, expression.length() - 1));
        }

        // Find the operator with the lowest precedence and use it as a root for the binary tree.
        // Then split the expression around the operator and handle these recursively.

        SearchResult match = nonBracketedIndexOf(expression, PatternLib.CONTAINSORPATTERN);
        if (match != null) {
            return new ORBinaryNode(parse(expression.substring(0, match.getStart() - 1)), parse(expression.substring(match.getEnd(),
                    expression.length())));
        }

        match = nonBracketedIndexOf(expression, PatternLib.CONTAINSANDPATTERN);
        if (match != null) {
            return new ANDBinaryNode(parse(expression.substring(0, match.getStart() - 1)), parse(expression.substring(match.getEnd(),
                    expression.length())));
        }

        match = nonBracketedIndexOf(expression, PatternLib.STARTSWITHNOTPATTERN);
        if (match != null && match.getStart() == 0) {
            return new NotNode(parse(expression.substring(match.getEnd(), expression.length())));
        }

        if (PatternLib.ENTIRENUMBERPATTERN.matcher(expression).matches()) {
            return new InputNode(Integer.valueOf(expression));
        } else if (expression.equalsIgnoreCase("true")) {
            return new ConstantNode(true);
        } else if (expression.equalsIgnoreCase("false")) {
            return new ConstantNode(false);
        } else {
            throw new IllegalArgumentException(
                    "Input isn't a valid boolean function definition. Please consult the javadoc for a detailed description.");
        }
    }

    /**
     * Returns the first occurrence of the needle in the haystack that is not surrounded by brackets
     * 
     * @param haystack
     *            The string where to search
     * @param needle
     *            The string to be searched
     * @return First non-bracketed occurrence
     */
    private static SearchResult nonBracketedIndexOf(String haystack, Pattern needle) {
        Matcher m = needle.matcher(haystack);
        while (m.find()) {
            // Make sure that it's not in brackets
            if (countBrackets(haystack, true, 0, m.start()) == countBrackets(haystack, false, 0, m.start())) {
                return new SearchResult(m.start(), m.end());
            }
        }
        return null;
    }

    /**
     * Counts the number of brackets in a string.
     * 
     * @param haystack
     *            The string where to search
     * @param opening
     *            Whether to search for opening (true) or closing brackets (false)
     * @param start
     *            Start and
     * @param end
     *            End of the region where to search
     * @return Number of brackets
     */
    private static int countBrackets(String haystack, boolean opening, int start, int end) {
        int counter = 0;
        for (int i = start; i <= end; i++) {
            if (opening) {
                if (haystack.charAt(i) == '(') {
                    counter++;
                }
            } else {
                if (haystack.charAt(i) == ')') {
                    counter++;
                }
            }
        }

        return counter;
    }

    @Override
    public boolean eval(boolean[] inputs) {
        if (root.isDisabled()) {
            return false;
        }

        return root.eval(inputs);
    }

    @Override
    public double interpolateSQUAD(double[] input, double[] squadWeights)  {
        throw new InterpolationMethodNotSupported();
    }

    @Override
    public double interpolateBooleCube(double[] input) {
        if (root.isDisabled()) {
            return 0;
        }

        return root.interpolateBooleCube(input);
    }

    @Override
    public TreeBooleanFunction clone() {
        return new TreeBooleanFunction(root.clone(), arity);
    }

    @Override
    protected BDD createBDDUnchecked(BDD[] inputs, BDDFactory bddFactory) {
         if (root.isDisabled()) {
         return bddFactory.zero();
         }
        return root.createBDD(inputs, bddFactory);
    }

    @Override
    public String toString() {
        return root.toString();
    }

    @Override
    public void disableConnection(int position) {
        root.disableConnection(position);
    }

    @Override
    public void restore() {
        root = originalTree.clone();
    }

    @Override
    public String getFunctionString(String[] nodeNames, boolean odefyCompatible) {
        if (nodeNames.length != getArity()) {
            throw new IllegalArgumentException("The number of node names must correspond to the arity of the function.");
        }
        if (odefyCompatible && getArity() == 0) {
            return "<>";
        }
        
        if (root.isDisabled()) {
        return "false";
        }

        return root.getFunctionString(nodeNames, odefyCompatible);
    }

}
