package jimena.calculationparameters;

import jimena.simulation.CalculationController;
import jimena.simulationmethods.SimulationMethod;

/**
 * A data structure with parameters for the search for stable states in a continuous model.
 */
public class ConvergenceParameters extends SSSimulationParameters {
    private int threads;
    private double duplicateMaxDiff;

    /**
     * Creates a new data structure with parameters for the search for a stable state starting from a given configuration in a continuous
     * model.
     * 
     * @param method
     *            The method to apply for the simulation
     * @param dt
     *            The time step in the simulation
     * @param maxT
     *            The maximum simulated time
     * @param calculationController
     *            A calculation controller or null for an uncontrolled calculation
     * @param stabilityMaxDiff
     *            The epsilon neighborhood around a former value which must not be left to assume stability has the diameter
     *            stabilityMaxDiff*2
     * @param stabilityMinTime
     *            Time the node has to spend in an epsilon neighborhood around a former value to assume stability
     * @param threads
     *            The number of threads used for the simulation
     * @param duplicateMaxDiff
     *            The maximum difference for two stable state to be considered equal
     */
    public ConvergenceParameters(SimulationMethod method, double dt, double maxT, CalculationController calculationController,
            double stabilityMaxDiff, double stabilityMinTime, int threads, double duplicateMaxDiff) {
        super(method, dt, maxT, calculationController, stabilityMaxDiff, stabilityMinTime);
        setThreads(threads);
        setDuplicateMaxDiff(duplicateMaxDiff);
    }

    /**
     * Returns the number of threads used for the simulation.
     * 
     * @return The number of threads used for the simulation
     */
    public int getThreads() {
        return threads;
    }

    /**
     * Sets the number of threads used for the simulation.
     * 
     * @param threads
     *            The number of threads used for the simulation
     */
    public void setThreads(int threads) {
        if (threads <= 0) {
            throw new IllegalArgumentException("The number of threads must be greaten than 0.");
        }

        this.threads = threads;
    }

    /**
     * Returns the maximum difference for two components to be considered equal.
     * 
     * @return The maximum difference for two components to be considered equal.
     */
    public double getDuplicateMaxDiff() {
        return duplicateMaxDiff;
    }

    /**
     * Sets the maximum difference for two components to be considered equal.
     * 
     * @param duplicateMaxDiff
     *            The maximum difference for two components to be considered equal to be set.
     */
    public void setDuplicateMaxDiff(double duplicateMaxDiff) {
        if (duplicateMaxDiff < 2 * getStabilityMaxDiff()) {
            throw new IllegalArgumentException(
                    "The maximum difference for two components to be considered equal may not be less then 2 times the stabilityMinDiff.");
        }

        this.duplicateMaxDiff = duplicateMaxDiff;
    }
    
    /**
     * Copies the data without the calculation controller.
     * 
     * @return A copy of the data.
     */
    public ConvergenceParameters cloneWithoutController() {
        return new ConvergenceParameters(method.clone(), dt, maxT, null, stabilityMaxDiff, stabilityMinTime, threads, duplicateMaxDiff);
    }
}
