package jimena.libs;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * A library with functions for arrays
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class ArrayLib {

    /**
     * Redefines a list of byte arrays to a list of object arrays.
     * 
     * @param data
     *            List of byte arrays
     * @return List of object arrays
     */
    public static ArrayList<Object[]> bytesListToObjectsListUnchecked(ArrayList<byte[]> data) {
        ArrayList<Object[]> result = new ArrayList<Object[]>();
        for (byte[] byteArray : data) {
            Object[] objectArray = new Object[byteArray.length];
            for (int i = 0; i < byteArray.length; i++) {
                objectArray[i] = byteArray[i];
            }
            result.add(objectArray);
        }
        return result;
    }

    /**
     * Redefines a list of double arrays to a list of object arrays.
     * 
     * @param data
     *            List of double arrays
     * @return List of object arrays
     */
    public static ArrayList<Object[]> doublesListToObjectsListUnchecked(List<double[]> data) {
        ArrayList<Object[]> result = new ArrayList<Object[]>();
        for (double[] byteArray : data) {
            Object[] objectArray = new Object[byteArray.length];
            for (int i = 0; i < byteArray.length; i++) {
                objectArray[i] = byteArray[i];
            }
            result.add(objectArray);
        }
        return result;
    }

    /**
     * Adds a new element to an array.
     * 
     * @param array
     *            The array to add an element to
     * @param newElement
     *            The new element
     * @return The new array
     */
    public static <T> T[] add(T[] array, T newElement) {
        if (array == null || newElement == null) {
            throw new NullPointerException();
        }

        T[] newArray = Arrays.copyOf(array, array.length + 1);
        newArray[array.length] = newElement;
        return newArray;
    }

    /**
     * Adds a new element to an array of doubles.
     * 
     * @param array
     *            The array to add an element to
     * @param newElement
     *            The new element
     * @return The new array
     */
    public static double[] add(double[] array, double newElement) {
        double[] newArray = Arrays.copyOf(array, array.length + 1);
        newArray[array.length] = newElement;
        return newArray;
    }

    /**
     * Adds a new element to an array of booleans.
     * 
     * @param array
     *            The array to add an element to
     * @param newElement
     *            The new element
     * @return The new array
     */
    public static boolean[] add(boolean[] array, boolean newElement) {
        boolean[] newArray = Arrays.copyOf(array, array.length + 1);
        newArray[array.length] = newElement;
        return newArray;
    }

    /**
     * Swaps two elements in a list
     * 
     * @param list
     *            The list in which elements are swap
     * @param a
     *            Index of the first element
     * @param b
     *            Index of the second element
     */
    public static <T> void swap(List<T> list, int a, int b) {
        // The code checks for null and out-of-bounds
        T temp = list.get(a);
        list.set(a, list.get(b));
        list.set(b, temp);
    }

    /**
     * Swaps two random elements in a list. It is not guaranteed that the swapped indexes out not identical, i.e. the list is not changed.
     * 
     * @param list
     *            The list in which elements are swap
     */
    public static <T> void swap(List<T> list) {
        int a = MathLib.getRandom().nextInt(list.size());
        int b = MathLib.getRandom().nextInt(list.size());
        swap(list, a, b); // The other swap function implicitly checks for null
    }
}
