package jimena.libs;

import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.TreeSet;

/**
 * A library of auxiliary mathematical functions.
 *
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 *
 */
public class MathLib {

    private static Random rng = new Random();

    public static Random getRandom() {
        return rng;
    }

    /**
     * Returns the set of all numbers from zero to a given maximum.
     *
     * @param max
     *            The maximum.
     * @return A treeset of the numbers.
     */
    public static TreeSet<Integer> numberSet(int max) {
        TreeSet<Integer> result = new TreeSet<Integer>();
        for (int i = 0; i <= max; i++) {
            result.add(i);
        }
        return result;
    }

    /**
     * Return the average of an array of double values.
     *
     * @param values
     *            The values to average.
     * @return Average of the values.
     */
    public static double average(double[] values) {
        if (values == null) {
            return Double.NaN;
        }

        double sum = 0;
        for (double value : values) {
            sum += value;
        }
        return sum / values.length;

    }

    /**
     * Return the average of selected items an array of double values.
     *
     * @param values
     *            The values to average.
     * @return Average of the values.
     */
    public static double average(double[] values, Set<Integer> valuesToConsider) {
        if (values == null || valuesToConsider == null || valuesToConsider.size() == 0) {
            return Double.NaN;
        }

        double sum = 0;
        for (Integer index : valuesToConsider) {
            sum += values[index];
        }
        return sum / valuesToConsider.size();

    }

    /**
     * Returns the minimum of two double values. If one value is NaN the second value is returned.
     *
     * @param d1
     *            The first value to compare.
     * @param d2
     *            The second value to compare.
     * @return The minimum.
     */
    public static double min(double d1, double d2) {
        if (Double.isNaN(d1)) {
            return d2;
        }
        if (Double.isNaN(d2)) {
            return d1;
        }
        return Math.min(d1, d2);
    }

    /**
     * Sums up an array of doubles.
     *
     * @param array
     *            The array to sum up.
     * @return The sum.
     */
    public static double sum(double[] array) {
        double sum = 0;
        for (double d : array) {
            sum += d;
        }
        return sum;
    }

    /**
     * Transforms an BDD allsat result (e.g. [-1, 0]) to an array of bytes (e.g. [[0, 0], [1, 0]]).
     *
     * @param bytes
     *            An allsat results
     * @return A bytes array representation of the result
     */
    public static ArrayList<byte[]> bddAllsatResultToArray(byte[] bytes) {
        if (bytes == null) {
            throw new NullPointerException();
        }
        for (byte b : bytes) {
            if (b > 1 || b < -1) {
                throw new IllegalArgumentException("The input is not a valid bdd allsat result.");
            }
        }

        ArrayList<byte[]> result = new ArrayList<byte[]>();
        result.add(new byte[bytes.length]);

        for (int i = 0; i < bytes.length; i++) {
            if (bytes[i] != -1) {
                for (byte[] resultBytes : result) {
                    resultBytes[i] = bytes[i];
                }
            } else {
                int oldSize = result.size();
                for (int j = 0; j < oldSize; j++) {
                    result.add(result.get(j).clone());
                }

                for (int j = result.size() / 2; j < result.size(); j++) {
                    result.get(j)[i] = 1;
                }
            }
        }

        return result;
    }

    /**
     * Transforms an array of BDD allsat results (e.g. [[1, 1],[-1, 0]]) to an array of bytes (e.g. [[1 ,1], [0, 0], [1, 0]]).
     *
     * @param list
     *            Array of BDD allsat results
     * @return A bytes array representation of the result
     */
    public static ArrayList<byte[]> bddAllsatResultToArray(List<byte[]> list) {
        if (list == null) {
            throw new NullPointerException();
        }

        for (byte[] bytes : list) {
            if (bytes == null) {
                continue;
            }
            for (byte b : bytes) {
                if (b > 1 || b < -1) {
                    throw new IllegalArgumentException("The input are not valid bdd allsat results.");
                }
            }
        }

        ArrayList<byte[]> result = new ArrayList<byte[]>();
        for (byte[] solution : list) {
            if (solution == null) {
                continue;
            }
            result.addAll(bddAllsatResultToArray(solution));
        }
        return result;
    }

    /**
     * Returns the mean squared difference of the elements of two arrays. The result is undefined/exception if array2.length !=
     * array1.length or if vector components are NaN.
     *
     * @param array1
     *            Array 1
     * @param array2
     *            Array 2
     * @return Mean squared difference
     */
    public static double meanSquaredDifferenceUnchecked(double[] array1, double[] array2) {
        return sumOfSquaredDifferenceUnchecked(array1, array2) / array1.length;
    }

    /**
     * Returns the mean squared difference of the elements of two arrays.
     *
     * @param array
     *            Array
     * @return Mean squared difference
     */
    public static double meanSquaredDifferenceUnchecked(double[] array) {
        return sumOfSquaredDifferenceUnchecked(array) / array.length;
    }

    /**
     * Returns the mean squared difference of the elements of two arrays. The result is undefined/exception if array2.length !=
     * array1.length or if vector components are NaN.
     *
     * @param array1
     *            Array 1
     * @param array2
     *            Array 2
     * @param componentsToConsider
     *            The vector components to consider
     * @return Mean squared difference
     */
    public static double meanSquaredDifferenceUnchecked(double[] array1, double[] array2, Set<Integer> componentsToConsider) {
        return sumOfSquaredDifferenceUnchecked(array1, array2, componentsToConsider) / componentsToConsider.size();
    }

    /**
     * Returns the mean squared difference of the elements of two arrays. The result is undefined/exception if array2.length !=
     * array1.length or if vector components are NaN.
     *
     * @param array1
     *            Array 1
     * @param array2
     *            Array 2
     * @return Mean squared difference
     */
    public static double meanSquaredDifferenceUnchecked(boolean[] array1, boolean[] array2) {
        return sumOfSquaredDifferenceUnchecked(array1, array2) / array1.length;
    }

    /**
     * Returns an array of the squared differences of the elements of two arrays.
     *
     * @param array1
     *            First input array.
     * @param array2
     *            Second input array.
     * @return The array of the squared differences of the elements of two arrays.F
     */
    public static double[] squaredDifferenceUnchecked(double[] array1, double[] array2) {
        double[] result = new double[array1.length];
        for (int i = 0; i < array1.length; i++) {
            result[i] = (array1[i] - array2[i]) * (array1[i] - array2[i]);
        }
        return result;
    }

    public static void invertSigns(double[] array) {
        for (int i = 0; i < array.length; i++) {
            array[i] = -array[i];
        }
    }

    /**
     * Returns an array of the squared differences of the elements of two arrays including the direction of the difference.
     *
     * @param array1
     *            First input array.
     * @param array2
     *            Second input array.
     * @return The array of the directional squared differences of the elements of two arrays.F
     */
    public static double[] squaredDirectionalDifferenceUnchecked(double[] array1, double[] array2) {
        double[] result = new double[array1.length];
        for (int i = 0; i < array1.length; i++) {
            result[i] = (array2[i] - array1[i]) * Math.abs(array2[i] - array1[i]);
        }
        return result;
    }

    /**
     * Returns the mean squared difference of the elements of two arrays. The result is undefined/exception if array2.length !=
     * array1.length or if vector components are NaN.
     *
     * @param array1
     *            Array 1
     * @param array2
     *            Array 2
     * @return Mean squared difference
     */
    public static double meanSquaredDifferenceUnchecked(byte[] array1, byte[] array2) {
        return sumOfSquaredDifferenceUnchecked(array1, array2) / array1.length;
    }

    /**
     * Returns the sum of the squared difference of the elements of two arrays. The result is undefined/exception if array2.length !=
     * array1.length or if vector components are NaN.
     *
     * @param array1
     *            Array 1
     * @param array2
     *            Array 2
     * @return Sum of the squared differences
     */
    public static double sumOfSquaredDifferenceUnchecked(double[] array1, double[] array2) {
        double result = 0;
        for (int i = 0; i < array1.length; i++) {
            result += (array1[i] - array2[i]) * (array1[i] - array2[i]);
        }

        return result;
    }

    /**
     * Returns the sum of the squared difference of the elements of an array and the null array.
     *
     * @param array
     *            An array
     * @return Sum of the squared differences
     */
    public static double sumOfSquaredDifferenceUnchecked(double[] array) {
        double result = 0;
        for (int i = 0; i < array.length; i++) {
            result += array[i] * array[i];
        }

        return result;
    }

    /**
     * Returns the sum of the squared difference of the elements of two arrays. The result is undefined/exception if array2.length !=
     * array1.length or if vector components are NaN.
     *
     * @param array1
     *            Array 1
     * @param array2
     *            Array 2
     * @param componentsToConsider
     *            The vector components to consider
     * @return Sum of the squared differences
     */
    public static double sumOfSquaredDifferenceUnchecked(double[] array1, double[] array2, Set<Integer> componentsToConsider) {
        double result = 0;
        for (int i = 0; i < array1.length; i++) {
            if (!componentsToConsider.contains(i)) {
                continue;
            }
            result += (array1[i] - array2[i]) * (array1[i] - array2[i]);
        }

        return result;
    }

    /**
     * Returns the sum of the squared difference of the elements of two arrays. The result is undefined/exception if array2.length !=
     * array1.length or if vector components are NaN.
     *
     * @param array1
     *            Array 1
     * @param array2
     *            Array 2
     * @return Sum of the squared differences
     */
    public static double sumOfSquaredDifferenceUnchecked(boolean[] array1, boolean[] array2) {
        double result = 0;
        for (int i = 0; i < array1.length; i++) {
            if (array1[i] != array2[i]) {
                result += 1;
            }
        }

        return result;
    }

    /**
     * Returns the sum of the squared difference of the elements of two arrays. The result is undefined/exception if array2.length !=
     * array1.length or if vector components are NaN.
     *
     * @param array1
     *            Array 1
     * @param array2
     *            Array 2
     * @return Sum of the squared differences
     */
    public static double sumOfSquaredDifferenceUnchecked(byte[] array1, byte[] array2) {
        double result = 0;
        for (int i = 0; i < array1.length; i++) {
            result += (array1[i] - array2[i]) * (array1[i] - array2[i]);
        }

        return result;
    }

    /**
     * Limits a value to a given range.
     *
     * @param value
     *            Value to limit
     * @param min
     *            Minimum value, the bound is ignored if it is NaN
     * @param max
     *            Maximum value, the bound is ignored if it is NaN
     * @return The limited value
     */
    public static double limitToRangeAndAssertNumerical(double value, double min, double max) {
        if (isNaN(value)) {
            throw new IllegalArgumentException("The given value is not a number.");
        }

        if (value < min) {
            return min;
        }

        if (value > max) {
            return max;
        }

        return value;
    }

    /**
     * Throws an IllegalArgumentException if the value is not between min and max or if the value ist NaN.
     *
     * @param value
     *            The value to inspect
     * @param min
     *            Minimum value
     * @param max
     *            Maximum value
     */
    public static void checkNotNaNAndWithinRange(double value, double min, double max) {
        if (value < min || value > max || isNaN(value)) {
            throw new IllegalArgumentException();
        }
    }

    /**
     * Throws an IllegalArgumentException if the value is not between min and max.
     *
     * @param value
     *            The value to inspect
     * @param min
     *            Minimum value
     * @param max
     *            Maximum value
     */
    public static void checkWithinRange(int value, int min, int max) {
        if (value < min || value > max) {
            throw new IllegalArgumentException();
        }
    }

    /**
     * Calculates the dot product of two factors. If one vector is shorter than the other the dot product will be calculated for the common
     * components. The result is undefined/exception if NaN/null values are given.
     *
     * @param f1
     *            Factor of the dot product
     * @param f2
     *            Factor of the dot product
     * @return Dot product
     */
    public static double dotProductUnchecked(double[] f1, double[] f2) {
        double sum = 0;

        for (int i = 0; i < f1.length; i++) {
            try {
                sum += f1[i] * f2[i];
            } catch (ArrayIndexOutOfBoundsException e) {
                // sum +=0
            }
        }

        return sum;
    }

    /**
     * Calculates the dot product of two factors. If one vector is shorter than the other the dot product will be calculated for the common
     * components. The result is undefined/exception if NaN/null values are given.
     *
     * @param f1
     *            First factor of the dot product
     * @param f2
     *            Second factor of the dot product
     * @param complement
     *            Whether the complement of f1 will be used or not
     * @return Dot product
     */
    public static double dotProductUnchecked(boolean[] f1, double[] f2, boolean complement) {
        double sum = 0;

        for (int i = 0; i < f1.length; i++) {
            try {
                if ((!complement) == f1[i]) {
                    sum += f2[i];
                }
            } catch (ArrayIndexOutOfBoundsException e) {
                // sum +=0
            }
        }

        return sum;
    }

    /**
     * Calculates the dot product of three factors. If one vector is shorter than the other the dot product will be calculated for the
     * common components. The result is undefined/exception if NaN/null values are given.
     *
     * @param f1
     *            First factor of the dot product
     * @param f2
     *            Second factor of the dot product
     * @param f3
     *            Third factor of the dot product
     * @param complement
     *            Whether the complement of f1 will be used or not
     * @return Dot product
     */
    public static double dotProductUnchecked(boolean[] f1, boolean[] f2, double[] f3, boolean complement) {
        double sum = 0;

        for (int i = 0; i < f1.length; i++) {
            try {
                if ((!complement) == f1[i] && !f2[i]) {
                    sum += f3[i];
                }
            } catch (ArrayIndexOutOfBoundsException e) {
                // sum +=0
            }
        }

        return sum;
    }

    /**
     * Calculates a ternary dot product of three factors. If one vector is shorter than the other the dot product will be calculated for the
     * common components. The result is undefined/exception if NaN/null values are given.
     *
     * @param f1
     *            First factor of the dot product
     * @param f2
     *            Second factor of the dot product
     * @param f3
     *            Third factor of the dot product
     * @param complement
     *            Whether the complement of f1 will be used or not
     * @return Dot product
     */
    public static double dotProductUnchecked(boolean[] f1, double[] f2, double[] f3, boolean complement) {
        double sum = 0;

        for (int i = 0; i < f1.length; i++) {
            try {
                if ((!complement) == f1[i]) {
                    sum += f2[i] * f3[i];
                }
            } catch (ArrayIndexOutOfBoundsException e) {
                // sum +=0
            }
        }

        return sum;
    }

    /**
     * Calculates a 4-ary dot product of four factors. If one vector is shorter than the other the dot product will be calculated for the
     * common components. The result is undefined/exception if NaN/null values are given.
     *
     * @param f1
     *            First factor of the dot product
     * @param f2
     *            Second factor of the dot product
     * @param f3
     *            Third factor of the dot product
     * @param f4
     *            Forth factor of the dot product
     * @param complement
     *            Whether the complement of f1 will be used or not
     * @return Dot product
     */
    public static double dotProductUnchecked(boolean[] f1, boolean[] f2, double[] f3, double[] f4, boolean complement) {
        double sum = 0;

        for (int i = 0; i < f1.length; i++) {
            try {
                if ((!complement) == f1[i] && !f2[i]) {
                    sum += f3[i] * f4[i];
                }
            } catch (ArrayIndexOutOfBoundsException e) {
                // sum +=0
            }
        }

        return sum;
    }

    /**
     * Adds two points. The result is undefined/exception if NaN/null values are given.
     *
     * @param p1
     *            First point
     * @param p2
     *            Second point
     * @return The sum
     */
    public static Point2D.Double addPointsUnchecked(Point2D.Double p1, Point2D.Double p2) {
        return new Point2D.Double(p1.getX() + p2.getX(), p1.getY() + p2.getY());
    }

    /**
     * Returns the center of a rectangle. The result is undefined/exception if NaN/null values are given.
     *
     * @param rect
     *            The rectangle whose center is to be determined
     * @return The center
     */
    public static Point2D.Double getRectCenterUnchecked(Rectangle2D.Double rect) {
        return new Point2D.Double(rect.getX() + rect.getWidth() / 2, rect.getY() + rect.getHeight() / 2);
    }

    /**
     * Calculate the intersection point of a rectangle and a line given by two points. The result is undefined/exception if NaN/null values
     * are given.
     *
     * @param r
     *            The rectangle
     * @param s
     *            First point of the line
     * @param t
     *            Second point of the line
     * @return The intersection point, or t if the line does not intersect the rectangle.
     */
    public static Point2D.Double lineIntersectRectUnchecked(Rectangle2D.Double r, Point2D.Double s, Point2D.Double t) {
        Line2D.Double line = new Line2D.Double(s, t);

        // Determine the side of the Rectangle that intersects the line
        if (line.intersectsLine(r.getX(), r.getY(), r.getX() + r.getWidth(), r.getY())) {
            // Top
            return intersectUnchecked(r.getX(), r.getY(), r.getX() + r.getWidth(), r.getY(), line);
        } else if (line.intersectsLine(r.getX(), r.getY(), r.getX(), r.getY() + r.getHeight())) {
            // Left
            return intersectUnchecked(r.getX(), r.getY(), r.getX(), r.getY() + r.getHeight(), line);
        } else if (line.intersectsLine(r.getX() + r.getWidth(), r.getY(), r.getX() + r.getWidth(), r.getY() + r.getHeight())) {
            // Right
            return intersectUnchecked(r.getX() + r.getWidth(), r.getY(), r.getX() + r.getWidth(), r.getY() + r.getHeight(), line);
        } else if (line.intersectsLine(r.getX(), r.getY() + r.getHeight(), r.getX() + r.getWidth(), r.getY() + r.getHeight())) {
            // Bottom
            return intersectUnchecked(r.getX(), r.getY() + r.getHeight(), r.getX() + r.getWidth(), r.getY() + r.getHeight(), line);
        } else {
            return t;
        }
    }

    /**
     * Intersects two line segments (or their extensions). If the extension of the line segments do not intersect or are identical null is
     * returned. The result is undefined/exception if NaN/null values are given.
     *
     * @param ax1
     *            X coordinate of the first point of line a.
     * @param ay1
     *            Y coordinate of the first point of line a.
     * @param ax2
     *            X coordinate of the second point of line a.
     * @param ay2
     *            Y coordinate of the second point of line a.
     * @param bx1
     *            X coordinate of the first point of line b.
     * @param by1
     *            Y coordinate of the first point of line b.
     * @param bx2
     *            X coordinate of the second point of line b.
     * @param by2
     *            Y coordinate of the second point of line b.
     * @return The intersection point or null.
     */
    public static Point2D.Double intersectUnchecked(double ax1, double ay1, double ax2, double ay2, double bx1, double by1, double bx2,
            double by2) {
        if (ax1 == ax2 && bx1 == bx2) {
            // Both lines a perpendicular. Either they do not intersect or they are identical
            return null;
        }

        if (ax1 == ax2) {
            // Line a is perpendicular
            double mb = (by1 - by2) / (bx1 - bx2);
            double tb = by1 - (mb * bx1);

            double x = ax1;
            double y = mb * x + tb;
            return new Point2D.Double(x, y);
        }

        if (bx1 == bx2) {
            // Line b is perpendicular
            double ma = (ay1 - ay2) / (ax1 - ax2);
            double ta = ay1 - (ma * ax1);

            double x = bx1;
            double y = ma * x + ta;

            return new Point2D.Double(x, y);
        }

        double ma = (ay1 - ay2) / (ax1 - ax2);
        double mb = (by1 - by2) / (bx1 - bx2);

        if (ma == mb) {
            // The lines are parallel
            return null;
        }

        // No special cases apply
        double ta = ay1 - (ma * ax1);
        double tb = by1 - (mb * bx1);

        double x = (tb - ta) / (ma - mb);
        double y = ma * x + ta;

        return new Point2D.Double(x, y);
    }

    /**
     * Intersects two line segments (or their extensions). If the extension of the line segments do not intersect or are identical null is
     * returned. The result is undefined/exception if NaN/null values are given.
     *
     * @param ax1
     *            X coordinate of the first point of line a.
     * @param ay1
     *            Y coordinate of the first point of line a.
     * @param ax2
     *            X coordinate of the second point of line a.
     * @param ay2
     *            Y coordinate of the second point of line a.
     * @param lineB
     *            Line b
     * @return The intersection point or null
     */
    public static Point2D.Double intersectUnchecked(double ax1, double ay1, double ax2, double ay2, Line2D.Double lineB) {
        return intersectUnchecked(ax1, ay1, ax2, ay2, lineB.getX1(), lineB.getY1(), lineB.getX2(), lineB.getY2());
    }

    /**
     * Intersects two line segments (or their extensions). If the extension of the line segments do not intersect or are identical null is
     * returned. The result is undefined/exception if NaN/null values are given.
     *
     * @param lineA
     *            Line a
     * @param lineB
     *            Line b
     * @return The intersection point or null
     */
    public static Point2D.Double intersectUnchecked(Line2D.Double lineA, Line2D.Double lineB) {
        return intersectUnchecked(lineA.getX1(), lineA.getY1(), lineA.getX2(), lineA.getY2(), lineB);
    }

    /**
     * Returns the length of a vector. The result is undefined/exception if NaN/null values are given.
     *
     * @param point
     *            The double array
     * @return The sum of the sqares
     */
    public static double lengthUnchecked(double[] point) {
        double result = 0;
        for (int i = 0; i < point.length; i++) {
            result += point[i] * point[i];
        }

        return Math.sqrt(result);
    }

    /**
     * Scales a vector. The result is undefined/exception if NaN/null values are given.
     *
     * @param vector
     *            The vector to scale
     * @param scale
     *            The scale factor
     */
    public static void scaleUnchecked(double[] vector, double scale) {
        for (int i = 0; i < vector.length; i++) {
            vector[i] *= scale;
        }
    }

    /**
     * Calculates the difference between two vectors. The result is undefined/exception if NaN/null values are given.
     *
     * @param minuend
     *            The minuend of the difference
     * @param subtrahend
     *            The subtrahend of the difference
     * @return The difference= minuend - subtrahend
     */
    public static double[] differenceUnchecked(double[] minuend, double[] subtrahend) {
        double[] result = new double[minuend.length];
        for (int i = 0; i < minuend.length; i++) {
            result[i] = minuend[i] - subtrahend[i];
        }
        return result;
    }

    /**
     * Returns vector with random values between 0 and 1.
     *
     * @param length
     *            Length of the vector
     * @return The random vector
     */
    public static double[] randomVector(int length) {
        if (length < 0) {
            throw new IllegalArgumentException("Can't create a random array of negative length.");
        }

        double[] result = new double[length];
        for (int i = 0; i < length; i++) {
            result[i] = Math.random();
        }
        return result;
    }

    /**
     * Returns vector with values 0, 0.5 or 1.
     *
     * @param length
     *            Length of the vector
     * @return The random vector
     */
    public static double[] randomVectorTernary(int length) {
        if (length < 0) {
            throw new IllegalArgumentException("Can't create a random array of negative length.");
        }

        double[] result = new double[length];
        for (int i = 0; i < length; i++) {
            double rnd = Math.random();
            if (rnd < ((double) 1) / 3) {
                result[i] = 0;
            } else if (rnd < ((double) 2) / 3) {
                result[i] = 0.5;
            } else {
                result[i] = 1;
            }

        }
        return result;
    }

    /**
     * Adds random vectors to a list of vectors.
     *
     * @param list
     *            The list to add to.
     * @param number
     *            Number of new vectors
     * @param length
     *            Length of the new vectors
     */
    public static void addRandomVectors(ArrayList<double[]> list, int number, int length) {
        if (length < 0) {
            throw new IllegalArgumentException("Can't create a random array of negative length.");
        }

        if (number < 0) {
            throw new IllegalArgumentException("Can't add a negative number of vector.");
        }

        for (int i = 0; i < number; i++) {
            list.add(randomVector(length));
        }
    }

    /**
     * Adds random ternary vectors to a list of vectors.
     *
     * @param list
     *            The list to add to.
     * @param number
     *            Number of new vectors
     * @param length
     *            Length of the new vectors
     */
    public static void addRandomVectorsTernary(ArrayList<double[]> list, int number, int length) {
        if (length < 0) {
            throw new IllegalArgumentException("Can't create a random array of negative length.");
        }

        if (number < 0) {
            throw new IllegalArgumentException("Can't add a negative number of vector.");
        }

        for (int i = 0; i < number; i++) {
            list.add(randomVectorTernary(length));
        }
    }

    /**
     * Returns a random item from several lists. If no elements are found an exception is thrown.
     *
     * @param lists
     *            Lists of vectors, at least one of them has to contain an element
     * @return A random element
     */
    @SuppressWarnings("unchecked")
    public static double[] randomElement(List<double[]>... lists) {
        int startList = MathLib.getRandom().nextInt(lists.length);
        for (int i = startList; i < lists.length + startList; i++) {
            if (lists[i % lists.length].size() == 0) {
                continue;
            }

            return lists[i % lists.length].get(MathLib.getRandom().nextInt(lists[i % lists.length].size()));
        }

        throw new IllegalArgumentException("No items found");
    }

    /**
     * Returns the mean vector from the vectors of several lists. The result is undefined/exception if NaN/null values are given, or if the
     * vectors do not have the same length.
     *
     * @param lists
     *            Lists of vectors, at least one list must contain an entry.
     * @return Mean vector
     */
    @SuppressWarnings("unchecked")
    public static double[] meanVectorUnchecked(ArrayList<double[]>... lists) {
        double[] result = null;
        double countedVectors = 0;

        for (ArrayList<double[]> list : lists) {
            for (double[] vector : list) {
                if (result == null) {
                    result = new double[vector.length];
                }
                addDoubleArraysUnchecked(result, vector);
                countedVectors++;
            }
        }

        scaleUnchecked(result, 1D / countedVectors);

        return result;
    }

    /**
     * Returns a random value between 0 and 1 that has a tendency to be far from a given value. The function is only defined for inputs
     * between 0 and 1.
     *
     * @param from
     *            The value to be far from
     * @return A value far from the given point
     */
    public static double far(double from) {
        while (true) {
            double candidate = Math.random();

            if (Math.random() < (candidate - from) * (candidate - from)) {
                return candidate;
            }
        }
    }

    /**
     * Returns a random value between 0 and 1 that has a tendency to be near a given value. The function is only defined for inputs between
     * 0 and 1.
     *
     * @param from
     *            The value to be close to
     * @return A value near the given point
     */
    public static double near(double from) {
        while (true) {
            double candidate = Math.random();
            double prob = (1 - (candidate - from) * (candidate - from));

            if (Math.random() < prob) {
                return candidate;
            }
        }
    }

    /**
     * Returns the maximum value from an array of double values.
     *
     * @param values
     *            The values
     * @return The maximum value
     */
    public static double max(double[] values) {
        double result = -Double.MAX_VALUE;
        for (double value : values) {
            result = Math.max(result, value);
        }
        return result;
    }

    /**
     * Transforms a byte array to a double array without checking the values.
     *
     * @param bytes
     *            The bytes array to transform
     * @return A doubles array with the same content
     */
    public static double[] transformByteArrayToDoubleArrayUnchecked(byte[] bytes) {
        double[] result = new double[bytes.length];
        for (int i = 0; i < bytes.length; i++) {
            result[i] = bytes[i];
        }
        return result;
    }

    /**
     * Checks whether a given (primitive) double value is NaN.
     *
     * @param value
     *            The value to check
     * @return true if it is NaN
     */
    public static boolean isNaN(double value) {
        return ((Double) value).isNaN();
    }

    /**
     * Checks whether a given (primitive) double value is NaN.
     *
     * @param value
     *            The value to check
     * @return false if it is NaN
     */
    public static boolean isNotNaN(double value) {
        return !isNaN(value);
    }

    /**
     * Adds array2 to array1.
     *
     * @param array1
     *            The array to add to.
     * @param array2
     *            The array to add.
     */
    public static void addDoubleArraysUnchecked(double[] array1, double[] array2) {
        for (int i = 0; i < array1.length; i++) {
            array1[i] = array1[i] + array2[i];
        }
    }

    /**
     * Divides the components of a double by a given value.
     *
     * @param array1
     *            The array to add to.
     * @param den
     *            The denominator.
     */
    public static void divideDoubleArraysUnchecked(double[] array1, double den) {
        for (int i = 0; i < array1.length; i++) {
            array1[i] = array1[i] / den;
        }
    }

    /**
     * Converts a byte array to a double array.
     *
     * @param array
     *            The array to convert.
     * @return A double array.
     */
    public static double[] byteArrayToDoubleArrayUnchecked(byte[] array) {
        double[] result = new double[array.length];
        for (int i = 0; i < array.length; i++) {
            result[i] = array[i];
        }
        return result;
    }

    /**
     * Converts a list of byte arrays to a list of double arrays.
     *
     * @param list
     *            The list to convert.
     * @return A list of double arrays.
     */
    public static ArrayList<double[]> byteArrayListToDoubleArrayListUnchecked(List<byte[]> list) {
        ArrayList<double[]> result = new ArrayList<double[]>();
        for (byte[] l : list) {
            result.add(byteArrayToDoubleArrayUnchecked(l));
        }
        return result;
    }

    public static int pow(int base, int exp) {
        int result = base;
        for (int i = 1; i < exp; i++) {
            result *= base;
        }
        return result;
    }

    public static double mean(List<Double> values) {
        double sum = 0;
        for (double d : values) {
            sum += d;
        }
        return sum / values.size();
    }

    /**
     * Returns the mean difference of values2 from values1
     *
     * @param values1
     *            First array of values
     * @param values2
     *            Second array of values
     * @return The mean relative difference
     */
    public static double meanDeviation(double[] values1, double[] values2) {
        double sum = 0;
        for (int i = 0; i < values1.length; i++) {
            sum += Math.abs(values1[i] - values2[i]);
        }
        return sum / values1.length;
    }

    public static double[] normalizeCentralities(double[] values) {
        double[] result = new double[values.length];
        for (int i = 0; i < values.length; i++) {
            result[i] = 1 / Math.pow(1 - Math.log10(Math.abs(values[i])), 3);
            if (values[i] < 0) {
                result[i] = -result[i];
            }
        }
        return result;
    }
}