package jimena.libs;

import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import jimena.sssearcher.SSSearchResult;

/**
 * A library of auxiliary string functions.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class StringLib {
    /**
     * Creates a comma separated string from a list of arrays. E.g. [[0, 1], [1, 2]] => "0.0, 1.0\n1.0, 2.0"
     * 
     * @param list
     *            List of double arrays
     * @return A comma separated string
     */
    public static String arrayToTabSeparatedUnchecked(ArrayList<Object[]> list) {
        StringBuilder result = new StringBuilder();
        for (int i = 0; i < list.size(); i++) {
            Object[] data = list.get(i);

            if (data.length != 0) {
                for (int j = 0; j < data.length; j++) {
                    if (j != 0) {
                        result.append("\t");
                    }
                    result.append(String.valueOf(data[j]));
                }
            }
            if (i != list.size() - 1) {
                result.append("\n");
            }
        }

        return result.toString();
    }

    /**
     * Creates a comma separated string from an array of arrays. E.g. [[0, 1], [1, 2]] => "0.0, 1.0\n1.0, 2.0"
     * 
     * @param list
     *            List of double arrays
     * @return A comma separated string
     */
    public static String arrayToCommaSeparatedUnchecked(byte[][] list) {
        StringBuilder result = new StringBuilder();
        for (int i = 0; i < list.length; i++) {
            byte[] data = list[i];

            if (data.length != 0) {
                for (int j = 0; j < data.length; j++) {
                    if (j != 0) {
                        result.append(", ");
                    }
                    result.append(String.valueOf(data[j]));
                }
            }
            if (i != list.length - 1) {
                result.append("\n");
            }
        }

        return result.toString();
    }

    /**
     * Writes a text to the clipboard.
     * 
     * @param content
     *            The String to write
     */
    public static void writeToClipboard(String content) {
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        Transferable transferable = new StringSelection(content);
        clipboard.setContents(transferable, null);
    }
    
    /**
     * Read the a string from the clipboard.
     * 
     * @return Any text found on the Clipboard; if none found, return an empty String.
     */
    static public String readFromClipboard() {
        String result = "";
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        // odd: the Object param of getContents is not currently used
        Transferable contents = clipboard.getContents(null);
        boolean hasTransferableText = (contents != null) && contents.isDataFlavorSupported(DataFlavor.stringFlavor);
        if (hasTransferableText) {
            try {
                result = (String) contents.getTransferData(DataFlavor.stringFlavor);
            } catch (UnsupportedFlavorException | IOException ex) {
                ex.printStackTrace();
            }
        }
        return result;
    }

    /**
     * Prints all vectors in a list.
     * 
     * @param list
     *            The list to print
     */
    public static void printDoubleVectorList(ArrayList<double[]> list) {
        if (list == null) {
            System.out.println("null");
            return;
        }
        for (double[] vector : list) {
            System.out.println(Arrays.toString(vector));
        }
    }

    /**
     * Prints the result of a SS search.
     * 
     * @param results
     *            Results of a SS search.
     * 
     */
    public static void printSSSearchResult(SSSearchResult results) {
        if (results == null) {
            System.out.println("null");
            return;
        }
        for (int i = 0; i < results.getResults().size(); i++) {
            System.out.println(results.getCounter().get(i) + ": " + Arrays.toString(results.getResults().get(i)));
        }
    }

    /**
     * Prints all vectors in a list.
     * 
     * @param list
     *            The list to print
     */
    public static void printByteVectorList(ArrayList<byte[]> list) {
        if (list == null) {
            System.out.println("null");
            return;
        }
        for (byte[] vector : list) {
            System.out.println(Arrays.toString(vector));
        }
    }

    /**
     * Returns whether two strings are identical ignoring white spaces at the beginning and at the end of each string.
     * 
     * @param one
     *            First string
     * @param two
     *            Second string
     * @return true if identical
     */
    public static boolean equalsTrimmed(String one, String two) {
        return one.trim().equals(two.trim());
    }

    /**
     * Prints SSs from a search in a readable format.
     * 
     * @param nodeNames
     *            Node names of the network.
     * @param ss
     *            SSs found in the network.
     */
    public static void printDiscreteSSS(String[] nodeNames, List<byte[]> ss) {
        int padding = maxStringLength(nodeNames);
        for (int i = 0; i < nodeNames.length; i++) {
            System.out.print(padLeft(nodeNames[i], padding));
            for (byte[] b : ss) {
                System.out.print(" " + b[i]);
            }
            System.out.println();
        }
    }

    /**
     * Adds spaces to the right of the string until a given length is reached.
     * 
     * @param string
     *            The string to be padded
     * @param length
     *            The desired length
     * @return The padded string
     */
    public static String padRight(String string, int length) {
        return String.format("%1$-" + length + "s", string);
    }

    /**
     * Adds spaces to the left of the string until a given length is reached.
     * 
     * @param string
     *            The string to be padded
     * @param length
     *            The desired length
     * @return The padded string
     */
    public static String padLeft(String string, int length) {
        return String.format("%1$" + length + "s", string);
    }

    /**
     * Returns the length of the longest string of a set of strings.
     * 
     * @param strings
     *            The set of strings to search in
     * @return Length of the longest string
     */
    public static int maxStringLength(String[] strings) {
        int max = 0;
        for (String s : strings) {
            max = Math.max(max, s.length());
        }
        return max;
    }
}
