package jimena.perturbation;


import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.event.TableModelEvent;

import jimena.binaryrn.NetworkNode;
import jimena.binaryrn.RegulatoryNetwork;

/**
 * Implements a window to display the perturbations of the network.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class PerturbationsWindow extends JFrame {
    private static final long serialVersionUID = 2436247424568506717L;
    private RegulatoryNetwork network;
    private JTable table;
    private PerturbationsModel model;

    /**
     * Create a new window that displays the perturbations.
     * 
     * @param network
     *            The network which contains the perturbations.
     */
    public PerturbationsWindow(RegulatoryNetwork network) {
        super("Perturbations Table");
        setIconImage(new ImageIcon("images" + File.separator + "settings16.png").getImage());

        this.network = network;

        setMinimumSize(new Dimension(450, 10));
        getContentPane().setLayout(new BorderLayout());

        model = new PerturbationsModel(network);

        // Show the perturbations of the network in a list
        table = new JTable(model) {
            private static final long serialVersionUID = 2476811052242836099L;

            @Override
            public Dimension getPreferredScrollableViewportSize() {
                return getPreferredSize();
            }

            @Override
            public void tableChanged(TableModelEvent e) {
                super.tableChanged(e);
                pack();
            }
        };
        
        table.setToolTipText("Double-click to show perturbation details.");

        table.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent e) {
                if (e.getButton() == MouseEvent.BUTTON1 && e.getClickCount() == 2) {
                    // A double click shows the perturbations details
                    PerturbationsWindow.this.model.rowToPerturbation(table.rowAtPoint(e.getPoint())).getWindow().setVisible(true);
                    PerturbationsWindow.this.network.notifyObserversOfChangedValues();
                }
            }
        });

        table.setTableHeader(null);

        getContentPane().add(new JScrollPane(table), BorderLayout.CENTER);

        // Add a control panel with buttons
        JPanel controlPanel = new JPanel(new GridLayout(1, 2));

        JButton addPerturbation = new JButton("Add New Perturbation");
        this.getRootPane().setDefaultButton(addPerturbation);

        addPerturbation.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                if (PerturbationsWindow.this.network.isEmpty()) {
                    return;
                }

                // Let the user choose the node
                NetworkNode node = (NetworkNode) JOptionPane.showInputDialog(PerturbationsWindow.this, "Node of the new perturbation: ",
                        "Choose the node", JOptionPane.PLAIN_MESSAGE, new ImageIcon(), PerturbationsWindow.this.network.getNetworkNodes(),
                        PerturbationsWindow.this.network.getNetworkNodes()[0]);
                if (node == null) {
                    return;
                }

                // Those are the perturbations offered to he user
                Perturbation[] perturbations = { new OnOffPerturbation(), new RandomPerturbation(), new SinePerturbation() };

                // Let the user choose the type of the perturbation
                Perturbation perturbation = (Perturbation) JOptionPane.showInputDialog(PerturbationsWindow.this,
                        "Type of the new perturbation: ", "Choose the type", JOptionPane.PLAIN_MESSAGE, new ImageIcon(), perturbations,
                        perturbations[0]);

                if (perturbation == null) {
                    return;
                }

                // Open the modal window and let the user set the parameters
                perturbation.getWindow().setVisible(true);

                if (!perturbation.creationWasAborted()) {
                    node.getPerturbations().add(perturbation);
                    PerturbationsWindow.this.network.notifyObserversOfChangedValues();
                }
            }
        });

        controlPanel.add(addPerturbation);

        JButton removePerturbation = new JButton("Remove Perturbation");

        removePerturbation.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                if (PerturbationsWindow.this.table.getSelectedColumn() != -1) {
                    model.deletePerturbation(PerturbationsWindow.this.table.getSelectedColumn());
                }

                PerturbationsWindow.this.network.notifyObserversOfChangedValues();
            }
        });

        controlPanel.add(removePerturbation);

        getContentPane().add(controlPanel, BorderLayout.PAGE_END);

        pack();        
        setResizable(false);
        setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    }
}
