package jimena.perturbation;

import jimena.libs.DoubleValue;
import jimena.libs.MathLib;

/**
 * Implements a perturbation which is inactive before a given onset and a takes random values after that until a given end.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class RandomPerturbation extends RealValuesPerturbation {
    private DoubleValue start;
    private DoubleValue end;
    private DoubleValue min;
    private DoubleValue max;
    private static final String[] valueNames = { "Start", "End", "Minimum", "Maximum" };

    /**
     * Creates a new perturbation which takes random values after an onset until a given end.
     * 
     * @param start
     *            Onset of the perturbation
     * @param end
     *            End of the perturbation
     * @param min
     *            Minimum value of the perturbation
     * @param max
     *            Maximum value of the perturbation
     */
    private RandomPerturbation(DoubleValue start, DoubleValue end, DoubleValue min, DoubleValue max) {
        // Checks are done by the DoubleValues, the DoubleValues cannot be null since the function is private
        super(new DoubleValue[] { start, end, min, max }, valueNames);
        this.start = start;
        this.end = end;
        this.min = min;
        this.max = max;
    }

    /**
     * Creates a new perturbation which takes random values after an onset until a given end.
     * 
     * If parameters are NaN, default values are chosen.
     * 
     * @param start
     *            Onset of the perturbation
     * @param end
     *            End of the perturbation
     * @param min
     *            Minimum value of the perturbation
     * @param max
     *            Maximum value of the perturbation
     */
    public RandomPerturbation(double start, double end, double min, double max) {
        // Checks are done by the DoubleValues
        this(new DoubleValue(MathLib.isNotNaN(start) ? start : 0), new DoubleValue(MathLib.isNotNaN(end) ? end : 1000), new DoubleValue(
                MathLib.isNotNaN(min) ? min : 0, 0, 1), new DoubleValue(MathLib.isNotNaN(max) ? max : 1, 0, 1));
    }

    /**
     * Creates a new perturbation which takes random values after an onset.
     * 
     * If parameters are NaN, default values are chosen.
     * 
     * @param onset
     *            Onset of the perturbation
     * @param min
     *            Minimum value of the perturbation
     * @param max
     *            Maximum value of the perturbation
     */
    public RandomPerturbation(double onset, double min, double max) {
        // Checks are done by the DoubleValues
        this(onset, Double.NaN, min, max);
    }

    /**
     * Creates a new perturbation which takes random values.
     * 
     * If parameters are NaN, default values are chosen.
     * 
     * @param min
     *            Minimum value of the perturbation
     * @param max
     *            Maximum value of the perturbation
     */
    public RandomPerturbation(double min, double max) {
        // Checks are done by the DoubleValues
        this(Double.NaN, Double.NaN, min, max);
    }

    /**
     * Creates a new perturbation which takes random between 0 and 1 values after an onset.
     * 
     * If parameters are NaN, default values are chosen.
     * 
     * @param onset
     *            Onset of the perturbation
     */
    public RandomPerturbation(double onset) {
        // Checks are done by the DoubleValues
        this(onset, Double.NaN, Double.NaN, Double.NaN);
    }

    /**
     * Creates a new perturbation which takes random values between 0 and 1.
     */
    public RandomPerturbation() {
        this(0);
    }

    @Override
    public double getValue(double t) {
        if (t >= start.getValue() && t <= end.getValue()) {
            double value = Math.random();

            // Scale the value
            return min.getValue() + value * (max.getValue() - min.getValue());
        } else {
            return -1;
        }
    }

    @Override
    public String getDescription() {
        return "Random value between " + min + " and " + max + " from " + start + " to " + end;
    }

    @Override
    public String toString() {
        return "Random Perturbation";
    }

    @Override
    public Perturbation clone() {
        return new RandomPerturbation(start.getValue(), end.getValue(), min.getValue(), max.getValue());
    }

}
