package jimena.perturbation;

import jimena.libs.DoubleValue;
import jimena.libs.MathLib;

/**
 * Implements a perturbation which is inactive before a given onset and is a sine function after that until a given end.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class SinePerturbation extends RealValuesPerturbation {
    private DoubleValue start;
    private DoubleValue end;
    private DoubleValue min;
    private DoubleValue max;
    private DoubleValue phase;
    private DoubleValue freq;
    private static final String[] valueNames = { "Start", "End", "Minimum", "Maximum", "Angular frequency", "Phase" };

    /**
     * Creates a new perturbation which is a sine function after an onset until a given end.
     * 
     * @param start
     *            Onset of the perturbation
     * @param end
     *            End of the perturbation
     * @param min
     *            Minimum value of the perturbation
     * @param max
     *            Maximum value of the perturbation
     * @param freq
     *            Angular frequency of the perturbation
     * @param phase
     *            Phase of the sine function
     */
    private SinePerturbation(DoubleValue start, DoubleValue end, DoubleValue min, DoubleValue max, DoubleValue freq, DoubleValue phase) {
        // Checks are done by the DoubleValues, the DoubleValues cannot be null since the function is private
        super(new DoubleValue[] { start, end, min, max, freq, phase }, valueNames);
        this.start = start;
        this.end = end;
        this.min = min;
        this.max = max;
        this.phase = phase;
        this.freq = freq;
    }

    /**
     * Creates a new perturbation which is a sine function after an onset until a given end.
     * 
     * If parameters are NaN, default values are chosen.
     * 
     * @param start
     *            Onset of the perturbation
     * @param end
     *            End of the perturbation
     * @param min
     *            Minimum value of the perturbation
     * @param max
     *            Maximum value of the perturbation
     * @param freq
     *            Angular frequency of the perturbation
     * @param phase
     *            Phase of the sine function
     */
    public SinePerturbation(double start, double end, double min, double max, double freq, double phase) {
        // Checks are done by the DoubleValues
        this(new DoubleValue(MathLib.isNotNaN(start) ? start : 0), new DoubleValue(MathLib.isNotNaN(end) ? end : 1000), new DoubleValue(
                MathLib.isNotNaN(min) ? min : 0, 0, 1), new DoubleValue(MathLib.isNotNaN(max) ? max : 1, 0, 1), new DoubleValue(
                MathLib.isNotNaN(freq) ? freq : 10), new DoubleValue(MathLib.isNotNaN(phase) ? phase : 0));
    }

    /**
     * Creates a new perturbation which is a sine function after an onset until a given end.
     * 
     * If parameters are NaN, default values are chosen.
     * 
     * @param start
     *            Onset of the perturbation
     * @param end
     *            End of the perturbation
     * @param min
     *            Minimum value of the perturbation
     * @param max
     *            Maximum value of the perturbation
     */
    public SinePerturbation(double start, double end, double min, double max) {
        // Checks are done by the DoubleValues
        this(start, end, min, max, Double.NaN, Double.NaN);
    }

    /**
     * Creates a new perturbation which is a sine function after an onset.
     * 
     * If parameters are NaN, default values are chosen.
     * 
     * @param start
     *            Onset of the perturbation
     * @param min
     *            Minimum value of the perturbation
     * @param max
     *            Maximum value of the perturbation
     */
    public SinePerturbation(double start, double min, double max) {
        // Checks are done by the DoubleValues
        this(start, Double.NaN, min, max, Double.NaN, Double.NaN);
    }

    /**
     * Creates a new perturbation which is a sine function.
     * 
     * If parameters are NaN, default values are chosen.
     * 
     * @param min
     *            Minimum value of the perturbation
     * @param max
     *            Maximum value of the perturbation
     */
    public SinePerturbation(double min, double max) {
        this(Double.NaN, Double.NaN, min, max, Double.NaN, Double.NaN);
    }

    /**
     * Creates a new perturbation which is a sine function between 0 and 1 after an onset.
     * 
     * If parameters are NaN, default values are chosen.
     * 
     * @param start
     *            Onset of the perturbation
     */
    public SinePerturbation(double start) {
        this(start, Double.NaN, Double.NaN, Double.NaN, Double.NaN, Double.NaN);
    }

    /**
     * Creates a new perturbation which is a sine function between 0 and 1.
     */
    public SinePerturbation() {
        this(0);
    }

    @Override
    public double getValue(double t) {
        if (t >= start.getValue() && t <= end.getValue()) {
            double value = 0.5 + 0.5 * Math.sin(t * freq.getValue() + phase.getValue());

            // Scale the value
            return min.getValue() + value * (max.getValue() - min.getValue());
        } else {
            return -1;
        }
    }

    @Override
    public String getDescription() {
        return "Sine function proportional to sin(" + freq + "*t + " + phase + ") between " + min + " and " + max + " from " + start + " to "
                + end;
    }

    @Override
    public String toString() {
        return "Sine Perturbation";
    }

    @Override
    public Perturbation clone() {
        return new SinePerturbation(start.getValue(), end.getValue(), min.getValue(), max.getValue(), freq.getValue(), phase.getValue());
    }

}
