package jimena.settings;


import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;

import javax.swing.JTextField;

import jimena.libs.MathLib;

/**
 * Encapsulates Information about settings in this settings window.
 * 
 * This class represents a setting of type integer.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class IntegerSetting extends AbstractSetting<Integer> {
    /**
     * Creates a new Setting. Every setting will automatically be registered for the settings windows.
     * 
     * @param key
     *            Key of this setting a Preferences structure
     * @param label
     *            A short label of this setting
     * @param description
     *            The description of this setting, if null is given the label will be used
     * @param minValue
     *            The minimum value
     * @param maxValue
     *            The maximum value
     * @param defaultValue
     *            The default value
     */
    public IntegerSetting(String key, String label, String description, int minValue, int maxValue, int defaultValue) {
        super(key, label, description, minValue, maxValue, defaultValue);

        MathLib.checkWithinRange(defaultValue, minValue, maxValue);
    }

    @Override
    public Integer getValue() {
        return preferences.getInt(getKey(), defaultValue);
    }

    /**
     * A text field bound to this setting.
     * 
     * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
     * 
     */
    public class IntegerSettingsTextField extends SettingsTextField {
        private static final long serialVersionUID = 2600825281394479832L;

        /**
         * Creates a new text field for the setting.
         */
        public IntegerSettingsTextField() {
            setText(String.valueOf(IntegerSetting.this.getValue()));
            addFocusListener(new FocusAdapter() {
                @Override
                public void focusLost(FocusEvent arg0) {
                    try {
                        if (Integer.valueOf(IntegerSettingsTextField.this.getText()) < minValue) {
                            IntegerSettingsTextField.this.setText(String.valueOf(minValue));
                        }
                        if (Integer.valueOf(IntegerSettingsTextField.this.getText()) > maxValue) {
                            IntegerSettingsTextField.this.setText(String.valueOf(maxValue));
                        }

                    } catch (NumberFormatException e) {
                        IntegerSettingsTextField.this.setText(String.valueOf(defaultValue));
                    }

                    preferences.putInt(getKey(), Integer.valueOf(IntegerSettingsTextField.this.getText()));
                }
            });
        }

        @Override
        public void reset() {
            IntegerSettingsTextField.this.setText(String.valueOf(defaultValue));
            preferences.putInt(getKey(), Integer.valueOf(IntegerSettingsTextField.this.getText()));
        }
    }

    private SettingsTextField settingsTextField = new IntegerSettingsTextField();

    @Override
    public JTextField getTextField() {
        return settingsTextField;
    }

    @Override
    public void reset() {
        settingsTextField.reset();
    }
}
