package jimena.simulation;

import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JProgressBar;
import javax.swing.SwingConstants;

/**
 * Implements a window that displays information about the progress of a calculation and gives the user the possibility to abort the
 * calcuation.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class ProgressWindow extends JDialog implements CalculationController {
    private static final long serialVersionUID = 2463655132948378857L;
    private JProgressBar progressbar;
    private JButton abortButton;
    private long creationTime = System.currentTimeMillis();
    private static final String titleStart = "Calculation running - ";
    private boolean isOn = true;

    /**
     * Creates a new simulation control window.
     */
    public ProgressWindow() {
        super((JFrame) null, "Calculation running", false);

        setLayout(new FlowLayout());

        progressbar = new JProgressBar(SwingConstants.HORIZONTAL, 0, 10000);
        progressbar.setPreferredSize(new Dimension(400, 30));
        add(progressbar);
        abortButton = new JButton("Abort Calculation", new ImageIcon("images\\stop32.png"));
        abortButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent arg0) {
                isOn = false;
                ProgressWindow.this.setVisible(false);
            }
        });

        add(abortButton);

        // Closing the window is equivalent to pressing the stop button
        addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent arg0) {
                isOn = false;
            }
        });

        pack();

        setVisible(true);
    }

    /**
     * Notifies the controller of the progress of the simulation.
     * 
     * @param progress
     *            Progress of the simulation
     * @param stepsDone
     *            Steps calculated so far
     */
    public void setProgress(double progress, int stepsDone) {
        progressbar.setValue((int) (progress * 10000));
        long now = System.currentTimeMillis();
        long secondLeft = (long) (((1 - progress) / progress) * (now - creationTime) / 1000);
        String durationLeft = String.format("%d:%02d:%02d", secondLeft / 3600, (secondLeft % 3600) / 60, (secondLeft % 60));
        int stepsPerSec = (int) (((double) stepsDone * 1000) / (now - creationTime));
        setTitle(titleStart + ((int) (progress * 100)) + "% - Time left: " + durationLeft + " - " + stepsPerSec + "/s");
    }

    @Override
    public void setProgress(int stepsDone, int maxSteps) {
        setProgress(((double) stepsDone) / maxSteps, stepsDone);
    }

    @Override
    public boolean isOn() {
        return isOn;
    }

    @Override
    public void notifyCalculationFinished() {
        setVisible(false);
        this.dispose();
    }
}
