package jimena.simulationmethods;

import jimena.binaryrn.RegulatoryNetwork;

/**
 * Implements an abstract continuous simulation method.
 *
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 *
 */
public abstract class ContinuousSimulationMethod extends SimulationMethod {

    public double getDerivativeValue(RegulatoryNetwork network, int nodeIndex, double[] values) {
        if (network.getNetworkNodes()[nodeIndex].getCustomFunction() != null) {
            return network.getNetworkNodes()[nodeIndex].getCustomFunction().eval(getNamedInputs(network, values));
        } else {
            return getDerivativeValueInternal(network, nodeIndex, values);
        }
    }

    @Override
    public void stepValuesOnly(RegulatoryNetwork network, double dt) {

        final int nodeCount = network.size();

        // Calculate new values first, then submit all new values at once
        double[] ksum = new double[nodeCount];

        double[] results1 = new double[nodeCount];
        double[] results2 = new double[nodeCount];

        for (int i = 0; i < nodeCount; i++) {
            double k = dt * getDerivativeValue(network, i, network.getValues(results2));
            results1[i] = k / 2 + network.getNetworkNodes()[i].getValue();
            ksum[i] = k;
        }

        for (int i = 0; i < nodeCount; i++) {
            double k = dt * getDerivativeValue(network, i, results1); // Estimate goes here
            results2[i] = k / 2 + network.getNetworkNodes()[i].getValue(); // Real state goes here
            ksum[i] += 2 * k;
        }

        for (int i = 0; i < nodeCount; i++) {
            double k = dt * getDerivativeValue(network, i, results2);
            results1[i] = k + network.getNetworkNodes()[i].getValue();
            ksum[i] += 2 * k;
        }

        for (int i = 0; i < nodeCount; i++) {
            double k = dt * getDerivativeValue(network, i, results1);
            network.getNetworkNodes()[i].setValue((ksum[i] + k) / 6 + network.getNetworkNodes()[i].getValue());
        }

    }

    public void stepValuesOnlyOld(RegulatoryNetwork network, double dt) {
        // TODO: DELETE ggf.
        // Old, but modified

        // Calculate new values first, then submit all new values at once
        double[] k1 = new double[network.size()];
        double[] k2 = new double[network.size()];
        double[] k3 = new double[network.size()];
        double[] k4 = new double[network.size()];

        double[] results1 = new double[network.size()];
        double[] results2 = new double[network.size()];
        double[] results3 = new double[network.size()];

        for (int i = 0; i < network.size(); i++) {
            k1[i] = getDerivativeValue(network, i, network.getValues());
            results1[i] = k1[i] * dt / 2 + network.getNetworkNodes()[i].getValue();
        }

        for (int i = 0; i < network.size(); i++) {
            k2[i] = getDerivativeValue(network, i, results1); // Estimate goes here
            results2[i] = k2[i] * dt / 2 + network.getNetworkNodes()[i].getValue(); // Real state goes here
        }

        for (int i = 0; i < network.size(); i++) {
            k3[i] = getDerivativeValue(network, i, results2);
            results3[i] = k3[i] * dt + network.getNetworkNodes()[i].getValue();
        }

        for (int i = 0; i < network.size(); i++) {
            k4[i] = getDerivativeValue(network, i, results3);
            network.getNetworkNodes()[i].setValue((k1[i] + 2 * k2[i] + 2 * k3[i] + k4[i]) / 6 * dt
                    + network.getNetworkNodes()[i].getValue());
        }
    }

    /**
     * Returns the current derivative of the value of the given node in the network.
     *
     * @param network
     *            The network which contains the node
     * @param nodeIndex
     *            The index of the node
     * @param values
     *            The values to assume for the network
     * @return The current derivative
     */
    public abstract double getDerivativeValueInternal(RegulatoryNetwork network, int nodeIndex, double[] values);

    @Override
    public boolean isContinuous() {
        return true;
    }
}
