package jimena.simulationmethods;

import java.util.HashMap;

import jimena.binarybf.BinaryBooleanFunction;
import jimena.binaryrn.NetworkNode;
import jimena.binaryrn.RegulatoryNetwork;

/**
 * Implements an abstract simulation method.
 *
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 *
 */
public abstract class SimulationMethod {
    private final static double STATESTABILITYINDEXDT = 0.00001;

    /**
     * Updates only the values of the network and not the time index.
     *
     * @param network
     *            The network
     * @param dt
     *            The time step. Will be ignored for discrete simulations
     */
    protected abstract void stepValuesOnly(RegulatoryNetwork network, double dt);

    /**
     * Executes one step of the simulation method on the network.
     *
     * @param network
     *            The network
     * @param dt
     *            The time step. Will be ignored for discrete simulations
     */
    public void stepUpdateNetwork(RegulatoryNetwork network, double dt) {
        stepValuesOnly(network, dt);

        network.getTimeIndex().addValue(dt);
    }

    /**
     * Calculates an index that shows whether the current state is stable or not.
     *
     * Invoking this method for non-continuous models is not recommended.
     *
     * @param network
     *            The network to look at
     *
     * @return 0 for steady stable states, > 0 otherwise
     */
    public double stateStabilityIndex(RegulatoryNetwork network) {
        // Store the current state
        double[] state = network.getValues();

        // Perform a small simulation step
        stepValuesOnly(network, STATESTABILITYINDEXDT);

        // Calculate the derivative of the network with this simulation method
        double result = 0;
        for (int i = 0; i < network.size(); i++) {
            result += (state[i] - network.getNetworkNodes()[i].getValue()) * (state[i] - network.getNetworkNodes()[i].getValue())
                    / STATESTABILITYINDEXDT / STATESTABILITYINDEXDT;
        }

        result /= network.size();

        network.setValues(state);

        return result;
    }

    /**
     * Gets a vector of boolean representations of the inputs to a given node.
     *
     * @param network
     *            The networks to take the values from
     * @param nodeIndex
     *            The node whose inputs a retrieved
     * @return A vector of the inputs
     */
    protected final boolean[] getBooleanInputs(RegulatoryNetwork network, int nodeIndex) {
        // Constructing the input array for every call might seem slow, but in fact it's far from being the bottleneck

        NetworkNode node = network.getNetworkNodes()[nodeIndex];
        BinaryBooleanFunction function = node.getFunction();

        boolean[] inputs = new boolean[function.getArity()];
        for (int i = 0; i < inputs.length; i++) {
            inputs[i] = network.getNetworkNodes()[node.getConnections()[i].getSource()].getValue() < 0.5 ? false : true;
        }

        return inputs;
    }

    /**
     * Returns the inputs to the given nodes assuming the given values for the node.
     *
     * @param network
     *            The network
     * @param nodeIndex
     *            Index of node to get the inputs of
     * @param values
     *            the values to assume
     * @return An array of the inputs
     */
    protected final double[] getInputs(RegulatoryNetwork network, int nodeIndex, double[] values) {
        // Constructing the input array for every call might seem slow, but in fact it's far from being the bottleneck

        NetworkNode node = network.getNetworkNodes()[nodeIndex];
        BinaryBooleanFunction function = node.getFunction();

        double[] inputs = new double[function.getArity()];
        for (int i = 0; i < inputs.length; i++) {
            inputs[i] = values[node.getConnections()[i].getSource()];
        }
        return inputs;
    }

    /**
     * Return return a Hashmap with the values of all nodes in the network when specific node values are assumed.
     *
     * @param network
     *            The network
     * @param values
     *            The node values to assume
     * @return An array of the inputs
     */
    protected final HashMap<String, Double> getNamedInputs(RegulatoryNetwork network, double[] values) {
        // Constructing the input array for every call might seem slow, but in fact it's far from being the bottleneck

        HashMap<String, Double> result = new HashMap<String, Double>();

        for (int i = 0; i < network.size(); i++) {
            result.put(network.getNetworkNodes()[i].getName(), values[i]);
        }
        return result;
    }

    /**
     * Return the name of the interpolation method.
     *
     * @return Name of the interpolation method.
     */
    @Override
    public abstract String toString();

    /**
     * Returns whether this method is continuous, i.e. the nodes have values between 0 and 1.
     *
     * @return true if the method is continuous
     */
    public abstract boolean isContinuous();

    @Override
    public abstract SimulationMethod clone();

}
