package jimena.sssearcher;

import java.util.ArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import jimena.binaryrn.RegulatoryNetwork;
import jimena.calculationparameters.ConvergenceParameters;
import jimena.gui.main.JimenaExecutor;
import jimena.simulation.ConvergenceFromStartVector;
import jimena.simulation.ConvergenceResult;

/**
 * An abstract superclass for steady stable state searchers (SSS).
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public abstract class SSSearcher {
    /**
     * Searches the steady stable states of a network.
     * 
     * Not all parameters are used by all subclasses.
     * 
     * @param network
     *            The network to simulate
     * @param parameters
     *            The parameters of the simulation.
     * @param maxTime
     *            The maximum time to search in ms
     * @return The steady stable states found by the searcher or null if the calculation was aborted by the user e.g. by the progress window
     */
    public abstract SSSearchResult searchSSStates(RegulatoryNetwork network, ConvergenceParameters parameters, long maxTime);

    /**
     * Searches the steady stable states of a network.
     * 
     * Not all parameters are used by all subclasses.
     * 
     * @param network
     *            The network to simulate
     * @param numberOfStarts
     *            The number of start vectors to simulate
     * @param parameters
     *            The parameters of the simulation.
     * @return The steady stable states found by the searcher or null if the calculation was aborted by the user e.g. by the progress window
     */
    public abstract SSSearchResult searchSSStates(RegulatoryNetwork network, long numberOfStarts, ConvergenceParameters parameters);

    /**
     * Executes a search for stable steady states with given start vectors. The result may contain duplicates!
     * 
     * @param network
     *            The network to simulate
     * @param p
     *            The parameters to use for the search
     * @param startVectors
     *            The vectors to start from
     * @param maxTime
     *            Maximum time for this calculation in ms, note that this is not identical to the maxTime of the whole search
     * @return The stable steady states found, if the maxTime did not suffice an empty array will be returned
     */
    static protected ArrayList<double[]> executeSearch(RegulatoryNetwork network, ConvergenceParameters p, ArrayList<double[]> startVectors,
            long maxTime) {

        ArrayList<double[]> results = new ArrayList<double[]>();
        long start = System.currentTimeMillis();

        ArrayList<double[]> realStartVectors = new ArrayList<double[]>();

        // No simulated annealing is used
        for (double[] vector : startVectors) {
            realStartVectors.add(vector);
        }

        // Simulate all start vectors (which may have been run through a simulated annealing)
        JimenaExecutor threadPool = new JimenaExecutor(p.getThreads());

        ArrayList<Future<ConvergenceResult>> searchers = new ArrayList<Future<ConvergenceResult>>();

        for (int i = 0; i < realStartVectors.size(); i++) {
            
            searchers.add(threadPool.submit(network.stableStateSearcher(realStartVectors.get(i), p.cloneWithoutController())));
        }

        threadPool.shutdown();
        try {
            while ((System.currentTimeMillis() - start) <= maxTime && !threadPool.isTerminated()) {
                threadPool.awaitTermination(250, TimeUnit.MILLISECONDS);
            }
        } catch (InterruptedException e1) {
            throw new RuntimeException("Interrupted.");
        }

        for (int i = 0; i < searchers.size(); i++) {
            try {
                if (searchers.get(i).get() != null) {
                    results.add(searchers.get(i).get().getResult());
                }
            } catch (InterruptedException | ExecutionException e) {
                e.printStackTrace();
            }
        }

        return results;
    }

    /**
     * Adds a list of vectors to another list ignoring the duplicates.
     * 
     * @param searchResult
     *            A data structure for search results.
     * @param newItems
     *            The new items
     * @param duplicateMaxDiff
     *            Maximum difference for each component of a possible duplicate
     * @param count
     *            Whether to count the new SSS
     */
    protected void addWithoutDuplicates(SSSearchResult searchResult, ArrayList<double[]> newItems, double duplicateMaxDiff, boolean count) {
        for (double[] newItem : newItems) {
            addDuplicateFree(searchResult, newItem, duplicateMaxDiff, count);
        }
    }

    /**
     * Adds a search result to a list of search result ignoring the duplicates.
     * 
     * @param searchResult
     *            A data structure for search results.
     * @param newItem
     *            The new item
     * @param duplicateMaxDiff
     *            Maximum difference for each component of a possible duplicate
     * @param count
     *            Whether to count the new SSS
     */
    protected void addDuplicateFree(SSSearchResult searchResult, double[] newItem, double duplicateMaxDiff, boolean count) {
        for (int j = 0; j < searchResult.getResults().size(); j++) {
            double[] oldItem = searchResult.getResults().get(j);
            boolean isIdentical = true;
            for (int i = 0; i < newItem.length; i++) {
                if (Math.abs(newItem[i] - oldItem[i]) > duplicateMaxDiff) {
                    isIdentical = false;
                    break;
                }
            }
            if (isIdentical) {
                if (count) {
                    searchResult.getCounter().set(j, searchResult.getCounter().get(j) + 1);
                }
                return;
            }
        }

        if (count) {
            searchResult.getCounter().add(1);
        } else {
            searchResult.getCounter().add(0);
        }
        searchResult.getResults().add(newItem);
    }
}
