package jimena.sssearcher;

import java.util.ArrayList;

import jimena.binaryrn.RegulatoryNetwork;
import jimena.calculationparameters.ConvergenceParameters;
import jimena.libs.MathLib;

/**
 * An abstract superclass for steady stable state searchers that deduce new start vectors for the search from the the start vectors we used
 * already and the stable steady states found already.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public abstract class StepwiseSearcher extends SSSearcher {
    private boolean initializeWithDiscreteSSS = true;

    /**
     * Sets whether the search is initialized with the discrete stable states of the network. For some networks, the search for the discrete
     * stable states is not computationally feasible.
     * 
     * @param initializeWithDiscreteSSS
     *            True if the search is to be initialized with the discrete stable states.
     */
    public void setInitializeWithDiscreteSSS(boolean initializeWithDiscreteSSS) {
        this.initializeWithDiscreteSSS = initializeWithDiscreteSSS;
    }

    /**
     * Returns the next start vectors of the searcher based on the start vectors we used already and the stable steady states found already.
     * 
     * @param network
     *            The network to search SSS in
     * @param oldStarts
     *            Start vectors we used already, MAY NOT BE EMPTY
     * @param foundStableStates
     *            Stable steady states found already
     * @return Start vectors to use next
     */
    protected abstract ArrayList<double[]> getNextSteps(RegulatoryNetwork network, ArrayList<double[]> oldStarts,
            ArrayList<double[]> foundStableStates);

    @Override
    public SSSearchResult searchSSStates(RegulatoryNetwork network, long numberOfStarts, ConvergenceParameters parameters) {
        int stepsDone = 0;
        ArrayList<double[]> oldStarts = new ArrayList<double[]>();
        SSSearchResult foundStableStates = new SSSearchResult();

        if (initializeWithDiscreteSSS) {
            // Add the discrete sss as start points
            ArrayList<byte[]> discreteStableSteadyStates = network.discreteStableSteadyStates();

            for (byte[] bytes : discreteStableSteadyStates) {
                oldStarts.add(MathLib.transformByteArrayToDoubleArrayUnchecked(bytes));
            }

            stepsDone += oldStarts.size();
            ArrayList<double[]> initialResults = executeSearch(network, parameters, oldStarts, Long.MAX_VALUE);
            addWithoutDuplicates(foundStableStates, initialResults, parameters.getDuplicateMaxDiff(), false);
        }

        while (stepsDone < numberOfStarts) {
            ArrayList<double[]> newStarts = getNextSteps(network, oldStarts, foundStableStates.getResults());
            addWithoutDuplicates(foundStableStates, executeSearch(network, parameters, newStarts, Long.MAX_VALUE),
                    parameters.getDuplicateMaxDiff(), true);
            oldStarts.addAll(newStarts);
            stepsDone += newStarts.size();
            if (parameters.getCalculationController() != null) {
                if (!parameters.getCalculationController().isOn()) {
                    return null;
                }
                parameters.getCalculationController().setProgress(((double) stepsDone) / numberOfStarts, stepsDone);
            }
        }

        if (parameters.getCalculationController() != null) {
            parameters.getCalculationController().notifyCalculationFinished();
        }

        return foundStableStates;
    }

    @Override
    public SSSearchResult searchSSStates(RegulatoryNetwork network, ConvergenceParameters parameters, long maxTime) {
        long start = System.currentTimeMillis();
        int stepsDone = 0;
        ArrayList<double[]> oldStarts = new ArrayList<double[]>();
        SSSearchResult foundStableStates = new SSSearchResult();

        if (initializeWithDiscreteSSS) {
            // Add the discrete sss as start points
            ArrayList<byte[]> discreteStableSteadyStates = network.discreteStableSteadyStates();

            for (byte[] bytes : discreteStableSteadyStates) {
                oldStarts.add(MathLib.transformByteArrayToDoubleArrayUnchecked(bytes));
            }

            stepsDone += oldStarts.size();

            ArrayList<double[]> initialResults = executeSearch(network, parameters, oldStarts, maxTime + start - System.currentTimeMillis());
            addWithoutDuplicates(foundStableStates, initialResults, parameters.getDuplicateMaxDiff(), false);
        }

        while ((System.currentTimeMillis() - start) < maxTime) {
            ArrayList<double[]> newStarts = getNextSteps(network, oldStarts, foundStableStates.getResults());
            addWithoutDuplicates(foundStableStates,
                    executeSearch(network, parameters, newStarts, maxTime + start - System.currentTimeMillis()),
                    parameters.getDuplicateMaxDiff(), true);
            oldStarts.addAll(newStarts);
            stepsDone += newStarts.size();
            if (parameters.getCalculationController() != null) {
                if (!parameters.getCalculationController().isOn()) {
                    return null;
                }
                parameters.getCalculationController().setProgress(((double) (System.currentTimeMillis() - start)) / maxTime, stepsDone);
            }
        }

        if (parameters.getCalculationController() != null) {
            parameters.getCalculationController().notifyCalculationFinished();
        }

        return foundStableStates;
    }
}
