package jimena.analysis;

import java.util.ArrayList;
import java.util.Set;
import java.util.concurrent.Future;

import jimena.binaryrn.RegulatoryNetwork;
import jimena.calculationparameters.ConvergenceParameters;
import jimena.gui.main.JimenaExecutor;
import jimena.libs.MathLib;
import jimena.simulation.CalculationController;
import jimena.simulation.ConvergenceResult;
import jimena.simulation.StableSteadyState;

/**
 * A method to compare two networks with respect to their convergence behavior.
 *
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 *
 */
public class ConvergenceComparator {
    /**
     * Update the calculation controller.
     *
     * @param searchers1
     *            Runs for the first network.
     * @param searchers2
     *            Runs for the second network.
     * @param controller
     *            The calculation controller.
     */
    private static void setCalculationController(ArrayList<Future<ConvergenceResult>> searchers1,
            ArrayList<Future<ConvergenceResult>> searchers2, CalculationController controller) {
        int sum = 0;
        int done = 0;
        for (Future<ConvergenceResult> run : searchers1) {
            sum++;
            if (run.isDone()) {
                done++;
            }
        }
        for (Future<ConvergenceResult> run : searchers2) {
            sum++;
            if (run.isDone()) {
                done++;
            }
        }

        if (controller != null) {
            controller.setProgress(done, sum);
        }
    }

    /**
     * Returns the difference between two networks. The aspect which is considered for the comparison is given by the subclass.
     *
     * @param n1
     *            One of the classes to compare.
     * @param n2
     *            One of the classes to compare.
     * @return The difference between the two networks
     * @throws Exception
     */
    public static double[] differenceV(RegulatoryNetwork n1, RegulatoryNetwork n2, int simulations, ConvergenceParameters parameters)
            throws Exception {
        JimenaExecutor threadPool = new JimenaExecutor(parameters.getThreads());

        ArrayList<Future<ConvergenceResult>> searchers1 = new ArrayList<Future<ConvergenceResult>>();
        ArrayList<Future<ConvergenceResult>> searchers2 = new ArrayList<Future<ConvergenceResult>>();

        for (int i = 0; i < simulations; i++) {
            double[] startValues = MathLib.randomVector(n1.size());

            RegulatoryNetwork networkN1 = n1.cloneClean();
            RegulatoryNetwork networkN2 = n2.cloneClean();

            networkN1.setValues(startValues);
            networkN2.setValues(startValues);

            // Not using the steadyStableState method avoids copying the networks
            searchers1.add(threadPool.submit(new StableSteadyState(networkN1, parameters.cloneWithoutController())));
            searchers2.add(threadPool.submit(new StableSteadyState(networkN2, parameters.cloneWithoutController())));
        }

        // Wait for termination of all calculations
        threadPool.shutdown();

        while (!threadPool.isTerminated()) {
            try {
                Thread.sleep(100);
                if (parameters.getCalculationController() != null && !parameters.getCalculationController().isOn()) {
                    return null;
                }

                setCalculationController(searchers1, searchers2, parameters.getCalculationController());

            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        double sumOfDifference[] = new double[n1.size()];
        int counted = 0;
        for (int i = 0; i < simulations; i++) {
            if (searchers1.get(i).get() != null && searchers2.get(i).get() != null) {
                MathLib.addDoubleArraysUnchecked(sumOfDifference,
                        MathLib.squaredDifferenceUnchecked(searchers1.get(i).get().getResult(), searchers2.get(i).get().getResult()));
                counted++;
            }
        }

        MathLib.scaleUnchecked(sumOfDifference, 1d / counted);

        if (parameters.getCalculationController() != null) {
            parameters.getCalculationController().notifyCalculationFinished();
        }

        if (threadPool.getError() != null) {
            throw threadPool.getError();
        }

        return sumOfDifference;
    }

    /**
     * Returns the difference between two networks. The aspect which is considered for the comparison is given by the subclass.
     *
     * @param n1
     *            One of the classes to compare.
     * @param n2
     *            One of the classes to compare.
     * @return The difference between the two networks
     * @throws Exception
     */
    public static double difference(RegulatoryNetwork n1, RegulatoryNetwork n2, int simulations, ConvergenceParameters parameters)
            throws Exception {
        return difference(n1, n2, n1.setOfAllNodeIndices(), simulations, parameters);
    }

    /**
     * Returns the difference between two networks. The aspect which is considered for the comparison is given by the subclass.
     *
     * @param n1
     *            One of the networks to compare.
     * @param n2
     *            One of the networks to compare.
     * @param nodesToConsider
     *            The nodes to consider for the comparison
     * @return The difference between the two networks
     * @throws Exception
     */
    public static double difference(RegulatoryNetwork n1, RegulatoryNetwork n2, Set<Integer> nodesToConsider, int simulations,
            ConvergenceParameters parameters) throws Exception {
        double[] temp = differenceV(n1, n2, simulations, parameters);
        double result = 0;
        int counter = 0;
        for (Integer i : nodesToConsider) {
            result += temp[i];
            counter++;
        }
        return result / counter;
    }

}
