package jimena.analysis;

import java.util.ArrayList;
import java.util.Set;

import jimena.binaryrn.NetworkConnection;

/**
 * Represents a set of NetworksInputs.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class NetworkConnectionsSet {
    private ArrayList<NetworkConnection> usedInputs = new ArrayList<NetworkConnection>();

    /**
     * Adds a new connection to the set.
     * 
     * @param node
     *            Node of the connection.
     * @param connection
     *            Number of the connection.
     */
    public void add(int node, int connection) {
        add(new NetworkConnection(node, connection));
    }

    /**
     * Adds a new connection to the set.
     * 
     * @param connection
     *            The connection to add.
     */
    public void add(NetworkConnection connection) {
        if (!contains(connection)) {
            usedInputs.add(connection);
        }
    }

    /**
     * Adds multiple new connections to the set.
     * 
     * @param connections
     *            The connections to add.
     */
    public void addAll(Set<NetworkConnection> connections) {
        for (NetworkConnection connection : connections) {
            add(connection);
        }
    }

    /**
     * Removes a connection from the set.
     * 
     * @param connections
     *            The connection to remove.
     */
    public void remove(NetworkConnection connections) {
        NetworkConnection toRemove = null;
        for (NetworkConnection usedInput : usedInputs) {
            if (usedInput.getNode() == connections.getNode() && usedInput.getPosition() == connections.getPosition()) {
                toRemove = usedInput;
            }
        }
        usedInputs.remove(toRemove);
    }

    /**
     * Removes multiple connections from the set.
     * 
     * @param connections
     *            The connections to remove.
     */
    public void removeAll(Set<NetworkConnection> connections) {
        for (NetworkConnection connection : connections) {
            remove(connection);
        }
    }

    /**
     * Checks whether the set contains a given connection.
     * 
     * @param node
     *            Node of the connection.
     * @param connections
     *            Number of the connection.
     * @return True if the set contains the connection.
     */
    public boolean contains(int node, int connections) {
        for (NetworkConnection usedInput : usedInputs) {
            if (usedInput.getNode() == node && usedInput.getPosition() == connections) {
                return true;
            }
        }
        return false;
    }

    /**
     * Checks whether the set contains a given connection.
     * 
     * @param connections
     *            The connection to search for.
     * @return True if the set contains the connection.
     */
    public boolean contains(NetworkConnection connections) {
        return contains(connections.getNode(), connections.getPosition());
    }
}