package jimena.binarybf;

import java.io.Serializable;

import net.sf.javabdd.BDD;
import net.sf.javabdd.BDDFactory;

/**
 * Implements an abstract immutable binary boolean function.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public abstract class BinaryBooleanFunction implements Serializable {
    private static final long serialVersionUID = 1157583500831313750L;

    /**
     * Will be thrown if a boolean function does not support the requested interpolation method.
     * 
     */
    public class InterpolationMethodNotSupported extends RuntimeException {
        private static final long serialVersionUID = -4202385161308792480L;

        /**
         * This exception will eventually be thrown (and displayed) if the function/the network does not support the chosen simulation
         * method.
         */
        public InterpolationMethodNotSupported() {
            super("Your model contains a node that cannot be simulated with the specified interpolation method.\n"
                    + "Please choose a universal method like BooleCube or HillCube interpolation.");
        }
    }

    /**
     * Applies the boolean functions to the inputs. The input will not be checked.
     * 
     * The inputs to this functions WILL NOT BE CHECKED. The result is undefined if the input arrays do not correspond to the arity of the
     * function or if the components of the input vectors are not in the range specified by the model.
     * 
     * @param inputs
     *            Inputs to the boolean function
     * @return Result of the boolean function
     */
    public abstract boolean eval(boolean[] inputs);

    /**
     * Creates a BDD for this function.
     * 
     * @param inputs
     *            The BDDs of the inputs
     * @param bddFactory
     *            The BDD factory
     * @return BDD for the function
     */
    public BDD createBDD(BDD[] inputs, BDDFactory bddFactory) {
        if (inputs == null) {
            throw new NullPointerException();
        }

        if (getArity() != inputs.length) {
            throw new IllegalArgumentException("Number of inputs must correspond to arity of the function.");
        }

        for (BDD input : inputs) {
            if (input == null) {
                throw new NullPointerException();
            }
        }

        if (bddFactory == null) {
            throw new NullPointerException();
        }

        return createBDDUnchecked(inputs, bddFactory);
    }

    /**
     * Creates a BDD for this function without checking the inputs.
     * 
     * @param inputs
     *            The BDDs of the inputs
     * @param bddFactory
     *            The BDD factory
     * @return BDD for the function
     */
    protected abstract BDD createBDDUnchecked(BDD[] inputs, BDDFactory bddFactory);

    /**
     * Returns the arity of the boolean function.
     * 
     * @return Arity of the function
     */
    public abstract int getArity();

    /**
     * Interpolates the boolean function using the SQUAD algorithm. (Calculates the \omega_i)
     * 
     * The inputs to this functions WILL NOT BE CHECKED. The result is undefined if the input arrays do not correspond to the arity of the
     * function or if the components of the input vectors are not in the range specified by the model.
     * 
     * @param inputs
     *            Inputs to the interpolation algorithm
     * @param squadWeights
     *            Weights of the inputs
     * @return The interpolated result.
     */
    public abstract double interpolateSQUAD(double[] inputs, double[] squadWeights);

    /**
     * Interpolates the boolean function using Boole- or HillCubes.
     * 
     * The inputs to this functions WILL NOT BE CHECKED. The result is undefined if the input arrays do not correspond to the arity of the
     * function or if the components of the input vectors are not in the range specified by the model.
     * 
     * @param inputs
     *            Inputs to the interpolation algorithm
     * @param hill
     *            Whether a Hill- or BooleCube will be used
     * @param hillNormalize
     *            Whether the inputs will be normalized after if a hill cube is applied.
     * @param hillNs
     *            N parameters of the HillCube
     * @param hillKs
     *            K parameters of the HillCube
     * @return The interpolated result.
     */
    public double interpolateBooleOrHillCube(double[] inputs, boolean hill, boolean hillNormalize, double[] hillNs, double[] hillKs) {
        return interpolateBooleCube(hill ? hill(inputs, hillNormalize, hillNs, hillKs) : inputs);
    }

    /**
     * Interpolates the function using Boole- or HillCubes.
     * 
     * @param inputs
     *            Inputs to the function
     * @return Result of the interpolation
     */
    protected abstract double interpolateBooleCube(double[] inputs);

    /**
     * Applies the hill function on all of the input values.
     * 
     * @param inputs
     *            Inputs to the hill function
     * @param hillNormalize
     *            Whether the inputs will be normalized if a hill cube is applied
     * @param hillNs
     *            N parameters of the HillCube
     * @param hillKs
     *            K parameters of the HillCube
     * @return Results of the hill functions
     */
    protected double[] hill(double[] inputs, boolean hillNormalize, double[] hillNs, double[] hillKs) {
        // Vgl Krumsiek et al. 2010

        double[] result = new double[inputs.length];

        for (int i = 0; i < inputs.length; i++) {
            double n = hillNs[i];
            double k = hillKs[i];
            double xn = Math.pow(inputs[i], n);
            double kn = Math.pow(k, n);

            result[i] = xn / (xn + kn);

            if (hillNormalize) {
                result[i] /= (1 / (1 + kn));
            }
        }

        return result;
    }

    /**
     * Creates a deep copy of the function.
     * 
     * @return Deep copy of the function
     */
    @Override
    public abstract BinaryBooleanFunction clone();

    @Override
    public abstract String toString();

    /**
     * Disabled the inputs with the given position.
     * 
     * @param position
     */
    public abstract void disableConnection(int position);

    /**
     * Removes all mutations from the function.
     */
    public abstract void restore();

    /**
     * Returns a string representation of the function.
     * 
     * @param nodeNames
     *            Names of the nodes to this function
     * @param odefyCompatible
     *            Whether the output should be in an Odefy compatible format.
     * @return String representation of the function.
     */
    public abstract String getFunctionString(String[] nodeNames, boolean odefyCompatible);

}
