package jimena.binarybf.treebf;

import net.sf.javabdd.BDD;
import net.sf.javabdd.BDDFactory;

/**
 * A node in a boolean tree that represents an AND. This is much faster than a common implementation for all binary nodes using a switch for
 * the eval function.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class ANDBinaryNode extends BinaryNode {
    private static final long serialVersionUID = -1890836409857875885L;

    /**
     * Creates a new node that represents an AND.
     * 
     * @param leaf1
     *            First operand
     * @param leaf2
     *            Second operand
     */
    public ANDBinaryNode(TreeNode leaf1, TreeNode leaf2) {
        super(leaf1, leaf2);
    }

    /**
     * Creates a new node that represents an AND.
     * 
     * @param leaf1
     *            First operand
     * @param leaf2
     *            Second operand
     * @param disabled Whether the input is disabled or not.
     */
    public ANDBinaryNode(TreeNode leaf1, TreeNode leaf2, boolean disabled) {
        super(leaf1, leaf2, disabled);
    }

    @Override
    protected boolean evalNode(boolean a, boolean b) {
        if (leaf1.isDisabled()) {
            return b;
        }
        if (leaf2.isDisabled()) {
            return a;
        }
        if (leaf1.isDisabled() && leaf2.isDisabled()) {
            throw new ArithmeticException("A binary node with two disabled subtrees cannot be evaluated.");
        }

        return a && b;
    }

    @Override
    public double interpolateBooleCube(double[] inputs) {
        if (leaf1.isDisabled()) {
            return leaf2.interpolateBooleCube(inputs);
        }
        if (leaf2.isDisabled()) {
            return leaf1.interpolateBooleCube(inputs);
        }
        if (leaf1.isDisabled() && leaf2.isDisabled()) {
            throw new ArithmeticException("A binary node with two disabled subtrees cannot be evaluated.");
        }

        return leaf1.interpolateBooleCube(inputs) * leaf2.interpolateBooleCube(inputs);
    }

    @Override
    public ANDBinaryNode clone() {
        return new ANDBinaryNode(leaf1.clone(), leaf2.clone(), isDisabled());
    }

    @Override
    protected BDD createBDD(BDD[] inputs, BDDFactory bddFactory) {
        if (leaf1.isDisabled()) {
            return leaf2.createBDD(inputs, bddFactory);
        }
        if (leaf2.isDisabled()) {
            return leaf1.createBDD(inputs, bddFactory);
        }
        if (leaf1.isDisabled() && leaf2.isDisabled()) {
            throw new ArithmeticException("A binary node with two disabled subtrees cannot be evaluated.");
        }

        return leaf1.createBDD(inputs, bddFactory).and(leaf2.createBDD(inputs, bddFactory));
    }

    @Override
    public String toString() {
        return "(" + leaf1.toString() + " AND " + leaf2.toString() + ")";
    }

    @Override
    public String getFunctionString(String[] nodeNames, boolean odefyCompatible) {
        if (leaf1.isDisabled()) {
            return leaf2.getFunctionString(nodeNames, odefyCompatible);
        }
        if (leaf2.isDisabled()) {
            return leaf1.getFunctionString(nodeNames, odefyCompatible);
        }
        if (leaf1.isDisabled() && leaf2.isDisabled()) {
            throw new ArithmeticException("A binary node with two disabled subtrees cannot be evaluated.");
        }

        return ("(" + leaf1.getFunctionString(nodeNames, odefyCompatible) + (odefyCompatible ? " && " : " AND ")
                + leaf2.getFunctionString(nodeNames, odefyCompatible) + ")");
    }
}