package jimena.binarybf.treebf;

import java.util.LinkedList;

/**
 * An abstract 2-ary node in a tree representing a binary function.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public abstract class BinaryNode extends TreeNode {
    private static final long serialVersionUID = -3569782960854244159L;
    /**
     * The first operands.
     */
    protected TreeNode leaf1;
    /**
     * The second operands.
     */
    protected TreeNode leaf2;

    /**
     * Creates a new node.
     * 
     * @param leaf1
     *            First subnode
     * @param leaf2
     *            Second subnode
     */
    public BinaryNode(TreeNode leaf1, TreeNode leaf2) {
        if (leaf1 == null || leaf2 == null) {
            throw new NullPointerException();
        }

        this.leaf1 = leaf1;
        this.leaf2 = leaf2;

        if (isCyclic()) {
            throw new IllegalArgumentException("The tree of a boolean function must be a acyclic.");
        }
    }

    /**
     * Creates a new node.
     * 
     * @param leaf1
     *            First subnode
     * @param leaf2
     *            Second subnode
     * @param disabled
     *            Whether the node is disabled or not.
     */
    public BinaryNode(TreeNode leaf1, TreeNode leaf2, boolean disabled) {
        this(leaf1, leaf2);
        setDisabled(disabled);
    }

    @Override
    public boolean eval(boolean[] inputs) {
        return evalNode(leaf1.eval(inputs), leaf2.eval(inputs));
    }

    /**
     * Evaluates the function represented by the node. value() can be used to evaluate the whole subtree.
     * 
     * @param a
     *            First input
     * @param b
     *            Second input
     * @return Result of the logic gate
     */
    protected abstract boolean evalNode(boolean a, boolean b);

    @Override
    public int getMinArity() {
        return Math.max(leaf1.getMinArity(), leaf2.getMinArity());
    }

    @Override
    protected boolean isCyclic(LinkedList<TreeNode> predecessors) {
        if (predecessors.contains(this)) {
            return true;
        }

        predecessors.add(this);
        boolean result = leaf1.isCyclic(predecessors) || leaf2.isCyclic(predecessors);
        predecessors.removeLast();
        return result;
    }

    @Override
    public void disableConnection(int position) {
        leaf1.disableConnection(position);
        leaf2.disableConnection(position);
        setDisabled(leaf1.isDisabled() && leaf2.isDisabled());
    }
}
