package jimena.binarybf.treebf;

import java.util.LinkedList;

import net.sf.javabdd.BDD;
import net.sf.javabdd.BDDFactory;

/**
 * Represents an input node in a tree representing a binary function.
 * 
 * The node always returns the input to the boolean function at position 'position'. E.g. if position = 2 and the function is f(a,b,c) =
 * (...) then the node will return input c.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class InputNode extends TreeNode {
    private static final long serialVersionUID = 5141676157409773471L;
    private int position;

    /**
     * Creates an input node.
     * 
     * @param position
     *            Position of the input to represent.
     */
    public InputNode(int position) {
        this.position = position;
    }
    
    /**
     * Creates an input node.
     * 
     * @param position
     *            Position of the input to represent.
     * @param disabled Whether the input node is disabled or not.
     */
    public InputNode(int position, boolean disabled) {
        this.position = position;
        setDisabled(disabled);
    }

    @Override
    public double interpolateBooleCube(double[] inputs) {
        if (isDisabled()) {
            throw new ArithmeticException("A unary disabled node may not be evaluated.");
        }

        return inputs[position];
    }

    @Override
    public boolean eval(boolean[] inputs) {
        if (isDisabled()) {
            throw new ArithmeticException("A unary disabled node may not be evaluated.");
        }

        return inputs[position];
    }

    @Override
    public int getMinArity() {
        return position + 1;
    }

    @Override
    protected boolean isCyclic(LinkedList<TreeNode> predecessors) {
        return false;
    }

    @Override
    public TreeNode clone() {
        return new InputNode(position, isDisabled());
    }

    @Override
    protected BDD createBDD(BDD[] inputs, BDDFactory bddFactory) {
        if (isDisabled()) {
            throw new ArithmeticException("A unary disabled node may not be evaluated.");
        }
        
        return inputs[position];
    }

    @Override
    public String toString() {
        return String.valueOf(position);
    }

    /**
     * Returns the position of the input this input node returns.
     * 
     * @return The position value
     */
    public int getPosition() {
        return position;
    }

    @Override
    public void disableConnection(int position) {
        if (this.position == position) {
            setDisabled(true);
        }
    }

    @Override
    public String getFunctionString(String[] nodeNames, boolean odefyCompatible) {
        if (isDisabled()) {
            throw new ArithmeticException("A unary disabled node may not be evaluated.");
        }
        
        
        return nodeNames[position];
    }
}
