package jimena.binarybf.treebf;

import java.util.LinkedList;

import net.sf.javabdd.BDD;
import net.sf.javabdd.BDDFactory;

/**
 * A NOT node in a tree representing a binary function.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class NotNode extends TreeNode {
    private static final long serialVersionUID = 18801225881561981L;
    private TreeNode leaf;

    /**
     * Creates a NOT node.
     * 
     * @param leaf
     *            Subtree to negate.
     */
    public NotNode(TreeNode leaf) {
        if (leaf == null) {
            throw new NullPointerException();
        }
        this.leaf = leaf;

        if (isCyclic()) {
            throw new IllegalArgumentException("The tree of a boolean function must be a acyclic.");
        }
    }

    /**
     * Creates a NOT node.
     * 
     * @param leaf
     *            Subtree to negate.
     * @param disabled Whether the node is disabled or not.
     */
    public NotNode(TreeNode leaf, boolean disabled) {
        this(leaf);
        setDisabled(disabled);
    }

    @Override
    public double interpolateBooleCube(double[] inputs) {
        if (isDisabled()) {
            throw new ArithmeticException("A unary disabled node may not be evaluated.");
        }

        return 1 - leaf.interpolateBooleCube(inputs);
    }

    @Override
    public boolean eval(boolean[] inputs) {
        if (isDisabled()) {
            throw new ArithmeticException("A unary disabled node may not be evaluated.");
        }

        return !leaf.eval(inputs);
    }

    @Override
    public int getMinArity() {
        return leaf.getMinArity();
    }

    @Override
    protected boolean isCyclic(LinkedList<TreeNode> predecessors) {
        if (predecessors.contains(this)) {
            return true;
        }

        predecessors.add(this);
        boolean result = leaf.isCyclic(predecessors);
        predecessors.removeLast();
        return result;
    }

    @Override
    public NotNode clone() {
        return new NotNode(leaf.clone(), isDisabled());
    }

    @Override
    protected BDD createBDD(BDD[] inputs, BDDFactory bddFactory) {
        if (isDisabled()) {
            throw new ArithmeticException("A unary disabled node may not be evaluated.");
        }

        return leaf.createBDD(inputs, bddFactory).not();
    }

    @Override
    public String toString() {
        return "(" + "NOT " + leaf.toString() + ")";
    }

    @Override
    public void disableConnection(int position) {
        leaf.disableConnection(position);
        setDisabled(leaf.isDisabled());
    }

    @Override
    public String getFunctionString(String[] nodeNames, boolean odefyCompatible) {
        if (isDisabled()) {
            throw new ArithmeticException("A unary disabled node may not be evaluated.");
        }

        return (odefyCompatible ? "(~" : "(NOT ") + leaf.getFunctionString(nodeNames, odefyCompatible) + ")";
    }
}
