package jimena.binarybf.treebf;

import java.io.Serializable;
import java.util.LinkedList;

import net.sf.javabdd.BDD;
import net.sf.javabdd.BDDFactory;

/**
 * Abstract node in a tree representing a binary function.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public abstract class TreeNode implements Serializable {
    private static final long serialVersionUID = -1616827313638259122L;
    private boolean disabled = false;

    /**
     * Returns the BooleCube Interpolation of the subtree whose root is this node.
     * 
     * @param input
     *            Input values the interpolation function
     * @return Interpolation
     */
    public abstract double interpolateBooleCube(double[] input);

    /**
     * Returns the evaluation of the subtree whose root is this node.
     * 
     * @param input
     *            Input values to the boolean function
     * @return Result of the Subtree
     */
    public abstract boolean eval(boolean[] input);

    /**
     * Recursively calculates the arity of the subtree as given by the highest input position used.
     * 
     * @return Arity
     */
    public abstract int getMinArity();

    /**
     * Returns whether the subtree of this node contains any of the predecessors or is cyclic.
     * 
     * @param predecessors
     *            The predecessors to check for
     * @return True, iff the subtree of this node contains any of the predecessors or is cyclic
     */
    protected abstract boolean isCyclic(LinkedList<TreeNode> predecessors);

    /**
     * Returns whether the subtree of this node is cyclic.
     * 
     * @return true, iff the subtree of this node is cyclic
     */
    public final boolean isCyclic() {
        return isCyclic(new LinkedList<TreeNode>());
    }

    /**
     * Creates a new deep copy of the node and its subnodes.
     */
    @Override
    public abstract TreeNode clone();

    /**
     * Replaces all inputs nodes which are subnodes of this node and have the given position value with a constant false node. May not be
     * called on InputNodes.
     * 
     * @param position
     *            The position value of the inputs to replace.
     */
    public abstract void disableConnection(int position);

    /**
     * Check whether the whole subtree defined by this node is disabled.
     * 
     * @return True if the whole subtree defined by this node is disabled.
     */
    public boolean isDisabled() {
        return disabled;
    }

    /**
     * Sets whether the whole subtree defined by this node is disabled.
     * 
     * @param disabled
     *            True if the whole subtree defined by this node is disabled.
     */
    protected void setDisabled(boolean disabled) {
        this.disabled = disabled;
    }

    /**
     * Creates a BDD for the node.
     * 
     * @param inputs
     *            Inputs to the function
     * @param bddFactory
     *            The BDD factory used for the creation of the BDD
     * @return BDD for the node.
     */
    protected abstract BDD createBDD(BDD[] inputs, BDDFactory bddFactory);

    /**
     * Returns a string representation of the function.
     * 
     * @param nodeNames
     *            Names of the nodes to this function
     * @param odefyCompatible
     *            Whether the output should be in an Odefy compatible format.
     * @return String representation of the function.
     */
    public abstract String getFunctionString(String[] nodeNames, boolean odefyCompatible);

    @Override
    public abstract String toString();
}
