package jimena.binarybf.treebf;

import java.util.ArrayList;
import java.util.List;

/**
 * A library with functions to create (sub-)tree of tree boolean functions.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class TreeNodeLib {
    /**
     * Builds a tree of TreeNodes from a list of activating and inhibiting subtrees.
     * 
     * @param activators
     *            Activating subtrees
     * @param inhibitors
     *            Inhibiting subtrees
     * @return A tree that combines the activating and inhibiting subtrees
     */
    public static TreeNode getActInhibitTree(List<TreeNode> activators, List<TreeNode> inhibitors) {
        if (activators == null || inhibitors == null) {
            throw new NullPointerException("Provide two empty arrays instead of null to create a constant-false tree");
        }

        // The components are checked by the OR- and AND-tree functions.
        TreeNode activatorSubtree = getOrTree(activators);
        TreeNode inhibitorSubtree = getAndTree(getNegatedTrees(inhibitors));

        if (activators.size() == 0) {
            return inhibitorSubtree;
        }
        if (inhibitors.size() == 0) {
            return activatorSubtree;
        }

        return new ANDBinaryNode(activatorSubtree, inhibitorSubtree);
    }

    /**
     * Negates a list of trees.
     * 
     * @param treeNodes
     *            TreeNodes to negate
     * @return Negated TreeNodes
     */
    public static ArrayList<TreeNode> getNegatedTrees(List<TreeNode> treeNodes) {
        if (treeNodes == null) {
            throw new NullPointerException();
        }

        ArrayList<TreeNode> negatedNodes = new ArrayList<TreeNode>();

        for (TreeNode treeNode : treeNodes) {
            // The node constructor checks for null
            negatedNodes.add(new NotNode(treeNode));
        }

        return negatedNodes;
    }

    /**
     * Creates a tree of AND nodes from a list of given leaves. If the input list is empty the constant true node is returned.
     * 
     * @param treeNodes
     *            Leaves for the AND tree
     * @return The root of the AND tree or the constant true node if no nodes are given
     */
    public static TreeNode getAndTree(List<TreeNode> treeNodes) {
        if (treeNodes == null) {
            throw new NullPointerException("Provide an empty array instead of null to create the constant true tree");
        }

        if (treeNodes.size() == 0) {
            return new ConstantNode(true);
        }

        TreeNode currentRoot = treeNodes.get(0);

        for (int i = 1; i < treeNodes.size(); i++) {
            // The node constructor checks for null
            currentRoot = new ANDBinaryNode(currentRoot, treeNodes.get(i));
        }

        return currentRoot;
    }

    /**
     * Creates a tree of OR nodes from a list of given leaves. If the input list is empty the constant true(!) node is returned.
     * 
     * @param treeNodes
     *            Leaves for the OR tree
     * @return The root of the OR tree or the constant false node if no nodes are given
     */
    public static TreeNode getOrTree(List<TreeNode> treeNodes) {
        if (treeNodes == null) {
            throw new NullPointerException("Provide an empty array instead of null to create the constant false tree");
        }

        if (treeNodes.size() == 0) {
            return new ConstantNode(true);
        }

        TreeNode currentRoot = treeNodes.get(0);

        for (int i = 1; i < treeNodes.size(); i++) {
            // The node constructor checks for null
            currentRoot = new ORBinaryNode(currentRoot, treeNodes.get(i));
        }

        return currentRoot;
    }
}
