package jimena.calculationparameters;

import jimena.simulation.CalculationController;
import jimena.simulationmethods.SimulationMethod;

/**
 * Encapsulates the parameters for the search for a stable state starting from a given configuration in a continuous model.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class SSSimulationParameters extends SimulationParameters {
    protected double stabilityMaxDiff;
    protected double stabilityMinTime;

    /**
     * Creates a new data structure with parameters for the search for a stable state starting from a given configuration in a continuous
     * model.
     * 
     * @param method
     *            The method to apply for the simulation
     * @param dt
     *            The time step in the simulation
     * @param maxT
     *            The maximum simulated time
     * @param stabilityMaxDiff
     *            The epsilon neighborhood around a former value which must not be left to assume stability has the diameter
     *            stabilityMaxDiff*2
     * @param stabilityMinTime
     *            Time the node has to spend in an epsilon neighborhood around a former value to assume stability
     * @param calculationController
     *            A calculation controller or null for an uncontrolled calculation
     */
    public SSSimulationParameters(SimulationMethod method, double dt, double maxT,
            CalculationController calculationController, double stabilityMaxDiff, double stabilityMinTime) {
        super(method, dt, maxT, calculationController);
        setStabilityMaxDiff(stabilityMaxDiff);
        setStabilityMinTime(stabilityMinTime);
    }

    /**
     * The maximum deviation during a simulation for a value to be considered stable.
     * 
     * @return The maximum deviation during a simulation for a value to be considered stable.
     */
    public double getStabilityMaxDiff() {
        return stabilityMaxDiff;
    }

    /**
     * Sets the maximum deviation during a simulation for a value to be considered stable.
     * 
     * @param stabilityMaxDiff
     *            The maximum deviation during a simulation for a value to be considered stable.
     */
    public void setStabilityMaxDiff(double stabilityMaxDiff) {
        if (stabilityMaxDiff <= 0) {
            throw new IllegalArgumentException("The stabilityMaxDiff must be greaten than 0.");
        }
        this.stabilityMaxDiff = stabilityMaxDiff;
    }

    /**
     * Returns the time the node has to spend in an epsilon neighborhood around a former value to assume stability.
     * 
     * @return The time the node has to spend in an epsilon neighborhood around a former value to assume stability
     */
    public double getStabilityMinTime() {
        return stabilityMinTime;
    }

    /**
     * Sets the time the node has to spend in an epsilon neighborhood around a former value to assume stability.
     * 
     * @param stabilityMinTime
     *            The time the node has to spend in an epsilon neighborhood around a former value to assume stability
     */
    public void setStabilityMinTime(double stabilityMinTime) {
        if (stabilityMinTime <= 0) {
            throw new IllegalArgumentException("The stabilityMinTime must be greaten than 0.");
        }
        this.stabilityMinTime = stabilityMinTime;
    }
    
    /**
     * Copies the data without the calculation controller.
     * 
     * @return A copy of the data.
     */
    public SSSimulationParameters cloneWithoutController() {
        return new SSSimulationParameters(method.clone(), dt, maxT, null, stabilityMaxDiff, stabilityMinTime);
    }

}
