package jimena.calculationparameters;

import jimena.simulation.CalculationController;
import jimena.simulationmethods.SimulationMethod;

/**
 * Encapsulates the parameters of a simulation.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class SimulationParameters {
    protected SimulationMethod method;
    protected double dt;
    protected double maxT;
    protected CalculationController calculationController;

    /**
     * Creates a new simulation parameters data structure
     * 
     * @param method
     *            The method to apply for the simulation
     * @param dt
     *            The time step in the simulation
     * @param maxT
     *            The maximum simulated time
     * @param calculationController
     *            A calculation controller or null for an uncontrolled calculation
     */
    public SimulationParameters(SimulationMethod method, double dt, double maxT, CalculationController calculationController) {
        setMethod(method);
        setDt(dt);
        setMaxT(maxT);
        setCalculationController(calculationController);
    }

    /**
     * Returns the method to apply for the simulation.
     * 
     * @return The method to apply for the simulation
     */
    public SimulationMethod getMethod() {
        if (method == null) {
            throw new NullPointerException();
        }
        return method;
    }

    /**
     * Sets the method to apply for the simulation.
     * 
     * @param method
     *            The method to apply for the simulation
     */
    public void setMethod(SimulationMethod method) {
        this.method = method;
    }

    /**
     * Returns the time step in the simulation.
     * 
     * @return The time step in the simulation
     */
    public double getDt() {
        return dt;
    }

    /**
     * Sets the time step in the simulation.
     * 
     * @param dt
     *            The time step in the simulation
     */
    public void setDt(double dt) {
        if (dt <= 0) {
            throw new IllegalArgumentException("The step width may not be less than 0.");
        }

        this.dt = dt;
    }

    /**
     * Returns the maximum simulated time.
     * 
     * @return The maximum simulated time
     */
    public double getMaxT() {
        return maxT;
    }

    /**
     * Sets the maximum simulated time.
     * 
     * @param maxT
     *            The maximum simulated time
     */
    public void setMaxT(double maxT) {
        if (maxT <= 0) {
            throw new IllegalArgumentException("The simulated time value may not be less than 0.");
        }

        this.maxT = maxT;
    }

    /**
     * Returns the calculation controller.
     * 
     * @return The the calculation controller
     */
    public CalculationController getCalculationController() {
        return calculationController;
    }

    /**
     * Sets the calculation controller (null for an uncontrolled calculation).
     * 
     * @param calculationController
     *            The calculation controller to set
     */
    public void setCalculationController(CalculationController calculationController) {
        this.calculationController = calculationController;
    }

    /**
     * Copies the data without the calculation controller.
     * 
     * @return A copy of the data.
     */
    public SimulationParameters cloneWithoutController() {
        return new SimulationParameters(method.clone(), dt, maxT, null);
    }
}
