package jimena.gui.charts;


import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.geom.Path2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.text.NumberFormat;

import javax.swing.JPanel;

import jimena.binaryrn.NetworkNode;
import jimena.binaryrn.RegulatoryNetwork;
import jimena.binaryrn.RegulatoryNetworkObserver;

/**
 * A panel to display the logs of the nodes.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class ChartsPanel extends JPanel implements RegulatoryNetworkObserver {
    private static final long serialVersionUID = 1851853418701407907L;
    private static final Color colors[] = { new Color(0x990000), new Color(0xFF0000), new Color(0xFF6633), new Color(0xFFCC00),
            new Color(0xFFFF00), new Color(0x009900C), new Color(0x00FF00), new Color(0x00CCCC), new Color(0x0033FF), new Color(0x00FFFF),
            new Color(0x000099), new Color(0x660099), new Color(0xFF0099), new Color(0xFF66FF), new Color(0x999999) };
    private RegulatoryNetwork network;
    // Distance of the origin from left and bottom border
    private static final double BD = 30;
    private NodeList nodeList;

    /**
     * Creates a new ChartsPanel for the given network.
     * 
     * @param network
     *            The network whose logs are to be displayed
     * @param nodeList
     *            The nodes of the network which are to be displayed
     */
    public ChartsPanel(RegulatoryNetwork network, NodeList nodeList) {
        if (network == null || nodeList == null) {
            throw new NullPointerException();
        }

        this.network = network;
        network.addObserver(this);

        this.nodeList = nodeList;
    }

    @Override
    public void paintComponent(Graphics g) {
        // Redraw the charts

        // Empty window
        super.paintComponent(g);

        try {
            NetworkNode[] displayedNodes = nodeList.getSelectedNodes();

            Graphics2D gg = (Graphics2D) g;
            gg.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            gg.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);

            // White background
            gg.setColor(Color.WHITE);
            gg.fill(gg.getClip());

            if (network.isEmpty()) {
                return;
            }

            // This check might become outdated later on if the network is reset during this drawing. This situation is caught be
            // exceptions.
            if (network.getNetworkNodes()[0].getLog().size() < 2) {
                return;
            }

            // Determine highest time index on the abscissa
            double lastTimeIndex = network.getNetworkNodes()[0].getLog().get(network.getNetworkNodes()[0].getLog().size() - 1).getX();

            // A transformation which transforms time indices and values.
            // We transform the shapes instead of the Graphics2D so we don't distort the stroke
            AffineTransform transform = new AffineTransform();
            transform.translate(BD, this.getHeight() - BD);
            transform.scale((this.getWidth() - 2 * BD) / lastTimeIndex, -(this.getHeight() - 2 * BD));

            // Draw the axes
            gg.setColor(new Color(0x444444));
            gg.setStroke(new BasicStroke(1.2F));

            // A Path2D can be transformed more easily
            Path2D.Double axes = new Path2D.Double();
            axes.moveTo(lastTimeIndex, 0);
            axes.lineTo(0, 0);
            axes.lineTo(0, 1);
            axes.transform(transform);
            gg.draw(axes);

            // Draw the small lines on the axes
            Path2D.Double steps = new Path2D.Double();

            NumberFormat numberFormat = NumberFormat.getNumberInstance();
            numberFormat.setMaximumFractionDigits(3);

            double xStep = stepSize(this.getWidth() - 2 * BD, lastTimeIndex, 90);
            double xPos = 0;
            while (xPos + xStep <= lastTimeIndex) {
                // Display the small lines on the abscissa
                xPos = xPos + xStep;
                Point2D linePosition = transform.transform(new Point2D.Double(xPos, 0), null);

                steps.moveTo(linePosition.getX(), linePosition.getY() + -2);
                steps.lineTo(linePosition.getX(), linePosition.getY() + 2);

                // Display the description
                FontMetrics metrics = gg.getFontMetrics();
                double descriptionWidth = metrics.stringWidth(numberFormat.format(xPos));
                Point2D.Double position = new Point2D.Double(linePosition.getX() - descriptionWidth / 2, linePosition.getY() + 15);
                gg.drawString(numberFormat.format(xPos), (float) position.getX(), (float) position.getY());
            }
            gg.draw(steps);

            steps = new Path2D.Double();

            numberFormat.setMaximumFractionDigits(2);
            numberFormat.setMinimumFractionDigits(2);
            double yStep = stepSize(this.getHeight() - 2 * BD, 1, 90);
            double yPos = 0;
            while (yPos + yStep <= 1.0001) {
                // Display the small lines on the ordinate
                yPos = yPos + yStep;

                Point2D linePosition = transform.transform(new Point2D.Double(0, yPos), null);

                steps.moveTo(-2 + linePosition.getX(), yPos + linePosition.getY());
                steps.lineTo(2 + linePosition.getX(), yPos + linePosition.getY());

                // Display the description
                FontMetrics metrics = gg.getFontMetrics();
                double descriptionWidth = metrics.stringWidth(numberFormat.format(yPos));
                Point2D.Double position = new Point2D.Double(linePosition.getX() + (-16 - descriptionWidth / 2), linePosition.getY() + 3);
                gg.drawString(numberFormat.format(yPos), (float) position.getX(), (float) position.getY());
            }
            gg.draw(steps);

            // Get broadest name of a node
            double maxNodeNameWidth = 0;
            for (NetworkNode node : displayedNodes) {
                FontMetrics metrics = gg.getFontMetrics();
                double nodeNameWidth = metrics.stringWidth(node.getName());
                maxNodeNameWidth = Math.max(nodeNameWidth, maxNodeNameWidth);
            }

            // Draw the charts and the legend
            gg.setStroke(new BasicStroke(2F));

            int nodeIndex = 0;

            for (NetworkNode node : displayedNodes) {
                // Draw the chart
                gg.setColor(colors[nodeIndex % (colors.length)]);

                if (node.getLog().size() < 2) {
                    // Not enough values to draw a chart
                    continue;
                }

                Path2D.Double path = new Path2D.Double();
                path.moveTo(node.getLog().get(0).getX(), node.getLog().get(0).getY());

                // Do not draw every entry in the log, but only sufficiently many
                double logStep = logStep(node.getLog().size(), this.getWidth() - 2 * BD, 2);

                for (double i = 1; ((int) i) < node.getLog().size(); i += logStep) {
                    path.lineTo(node.getLog().get((int) i).getX(), node.getLog().get((int) i).getY());
                }

                path.transform(transform);

                gg.draw(path);
                nodeIndex++;
            }

            nodeIndex = 0;

            for (NetworkNode node : displayedNodes) {
                // Draw the legend of the node
                gg.setColor(colors[nodeIndex % (colors.length)]);
                gg.fill(new Rectangle2D.Double(this.getWidth() - maxNodeNameWidth - 40, 10 + 20 * nodeIndex, 10, 10));

                gg.setColor(Color.black);
                gg.drawString(node.getName(), (int) (this.getWidth() - maxNodeNameWidth - 22), 18 + 20 * nodeIndex);

                nodeIndex++;
            }

        } catch (IndexOutOfBoundsException e) {
            // This may happen e.g. if the simulation is reset during the drawing
            // We just show a white window (for a moment)
            Graphics2D gg = (Graphics2D) g;
            gg.setColor(Color.WHITE);
            gg.fill(gg.getClip());
        } catch (NullPointerException e) {
            // This may happen e.g. if the simulation is reset during the drawing
            // We just show a white window (for a moment)
            Graphics2D gg = (Graphics2D) g;
            gg.setColor(Color.WHITE);
            gg.fill(gg.getClip());
        }
    }

    @Override
    public void notifyNetworkChanged() {
        notifyValuesChanged();
    }

    @Override
    public void notifyValuesChanged() {
        this.repaint();
    }

    /**
     * Returns the logical step size on an axis from a given physical length, logical length and a limit for the maximum (physical) pixels
     * per step.
     * 
     * @param pixelLength
     *            The physical length of the axis
     * @param length
     *            The logical length of the axis
     * @param maxPixelPerStep
     *            The maximum physical length of the step
     * @return The logical step size
     */
    private static double stepSize(double pixelLength, double length, double maxPixelPerStep) {
        double logicalStep = 1000000;
        while (pixelLength / (length / logicalStep) > maxPixelPerStep) {
            logicalStep /= 10;
        }

        if (pixelLength / (length / (5 * logicalStep)) <= maxPixelPerStep) {
            return logicalStep * 5;
        }

        if (pixelLength / (length / (2 * logicalStep)) <= maxPixelPerStep) {
            return logicalStep * 2;
        }

        return logicalStep;

    }

    /**
     * Returns the mean number of log entries skipped between two points of the chart.
     * 
     * @param numberOfLogEntries
     *            Number of available log entries
     * @param abscissaPixelLength
     *            Length of the abscissa in pixels
     * @param abscissaPixelPerStep
     *            Pixels between two points of the chart
     * @return Mean number of log entries skipped
     */
    private static double logStep(double numberOfLogEntries, double abscissaPixelLength, double abscissaPixelPerStep) {
        return Math.max(numberOfLogEntries / (abscissaPixelLength / abscissaPixelPerStep), 1);
    }
}
