package jimena.gui.guilib;

import java.awt.Component;
import java.util.EventObject;

import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.event.CellEditorListener;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableModel;
import javax.swing.text.JTextComponent;

/**
 * Implements a JTable in which the content of a cell is selected if its editing is started.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 */
public class AutoselectingTable extends JTable {
    private static final long serialVersionUID = 3376568035196666900L;

    /**
     * A cell editor which replaces delegates all call to a given template. If the template returns a text component, all contents of the
     * field are selected.
     * 
     * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
     * 
     */
    class AutoselectingEditor implements TableCellEditor {
        private TableCellEditor template;

        /**
         * Creates a new autoselecting editor with a given template.
         * 
         * @param template
         *            The template to which the calls are delegated.
         */
        public AutoselectingEditor(TableCellEditor template) {
            this.template = template;
        }

        @Override
        public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
            Component component = template.getTableCellEditorComponent(table, value, isSelected, row, column);
            if (component instanceof JTextComponent) {
                final JTextComponent textComponent = (JTextComponent) component;
                textComponent.requestFocus();
                SwingUtilities.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        textComponent.selectAll();
                    }
                });

            }
            return component;
        }

        @Override
        public void addCellEditorListener(CellEditorListener arg0) {
            template.addCellEditorListener(arg0);
        }

        @Override
        public void cancelCellEditing() {
            template.cancelCellEditing();
        }

        @Override
        public Object getCellEditorValue() {
            return template.getCellEditorValue();
        }

        @Override
        public boolean isCellEditable(EventObject arg0) {
            return template.isCellEditable(arg0);
        }

        @Override
        public void removeCellEditorListener(CellEditorListener arg0) {
            template.removeCellEditorListener(arg0);
        }

        @Override
        public boolean shouldSelectCell(EventObject arg0) {
            return template.shouldSelectCell(arg0);
        }

        @Override
        public boolean stopCellEditing() {
            return template.stopCellEditing();
        }
    }

    /**
     * Creates a new autoselecting table from a given table model.
     * 
     * @param model
     *            The table model to use
     */
    public AutoselectingTable(TableModel model) {
        super(model);
    }

    @Override
    public TableCellEditor getCellEditor(int row, int column) {
        return new AutoselectingEditor(super.getCellEditor(row, column));
    }
}
