package jimena.gui.guilib;

import java.util.ArrayList;

import javax.swing.table.AbstractTableModel;

/**
 * A table model to display a rectangular array of data entries.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class DataTableModel extends AbstractTableModel {
    private static final long serialVersionUID = 4633388478606224862L;
    private ArrayList<Object[]> data;
    private String[] titles;
    private Class<?> type;
    private String[] horizontal;

    /**
     * Creates a new table.
     * 
     * @param data
     *            Data of the table
     * @param titles
     *            Names of the data arrays
     * @param type
     *            Type of the entries in the table model
     * @param horizontal
     *            Titles of the columns if the data is displayed horizontally, null for a vertical presentation
     */
    protected DataTableModel(ArrayList<Object[]> data, String[] titles, Class<?> type, String[] horizontal) {
        if (data == null || titles == null) {
            throw new NullPointerException();
        }

        // Check dimensions of the data
        if (horizontal != null) {
            if (data.size() != horizontal.length - 1) {
                throw new IllegalArgumentException("The data for the table model is not rectangular.");
            }

            for (Object[] objects : data) {
                if (objects.length != titles.length) {
                    throw new IllegalArgumentException("The data for the table model is not rectangular.");
                }
            }
        } else {
            for (Object[] objects : data) {
                if (objects.length != titles.length) {
                    throw new IllegalArgumentException("The data for the table model is not rectangular.");
                }
            }
        }

        // Check type of the data
        for (Object[] objects : data) {
            for (Object object : objects) {
                try {
                    type.cast(object);
                } catch (ClassCastException e) {
                    throw new IllegalArgumentException("Not all data given to the output window has the given type.");
                }
            }
        }

        this.data = data;
        this.titles = titles;
        this.type = type;
        this.horizontal = horizontal;
    }

    @Override
    public Class<?> getColumnClass(int columnIndex) {
        if (horizontal != null) {
            if (columnIndex == 0) {
                return String.class;
            } else {
                return type;
            }
        } else {
            return type;
        }
    }

    @Override
    public int getColumnCount() {
        if (horizontal != null) {
            return data.size() + 1;
        } else {
            return titles.length;
        }
    }

    @Override
    public String getColumnName(int columnIndex) {
        if (horizontal != null) {
            return horizontal[columnIndex];
        } else {
            return titles[columnIndex];
        }
    }

    @Override
    public int getRowCount() {
        if (horizontal != null) {
            return titles.length;
        } else {
            return data.size();
        }
    }

    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        if (horizontal != null) {
            if (columnIndex == 0) {
                return titles[rowIndex];
            } else {
                return data.get(columnIndex - 1)[rowIndex];
            }
        } else {
            return data.get(rowIndex)[columnIndex];
        }

    }

    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        return false;
    }
}
