package jimena.gui.main;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.RenderingHints;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.geom.AffineTransform;
import java.awt.geom.Line2D;
import java.awt.geom.NoninvertibleTransformException;
import java.awt.geom.Path2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.geom.RoundRectangle2D;
import java.awt.image.BufferedImage;

import javax.swing.JPanel;

import jimena.binarybf.actinhibitf.ActivatorInhibitorFunction;
import jimena.binaryrn.Connection;
import jimena.binaryrn.NetworkNode;
import jimena.binaryrn.RegulatoryNetwork;
import jimena.binaryrn.RegulatoryNetworkObserver;
import jimena.libs.MathLib;

/**
 * A panel to displays the network.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class PanelGraphDrawer extends JPanel implements RegulatoryNetworkObserver {
    private static final long serialVersionUID = 6827002870716796773L;
    private static final double NODEPADDING = 50;
    private static final double ARROWTIPANGLE = 0.45;
    private static final double ARROWTIPLENGTH = 13;
    private static final double NODERECTANGLEROUNDING = 10;
    private static final double NEGATINGARROWWIDTH = 9;
    private static final double NEGATINGARROWTHICKNESS = 5;

    private RegulatoryNetwork network = new RegulatoryNetwork();
    private BufferedImage background = new BufferedImage(1, 1, BufferedImage.TYPE_4BYTE_ABGR);
    private Graphics2D backgroundGraphics = (Graphics2D) background.getGraphics();

    private HasActivatableNodes leftClickListener;
    private HasActivatableNodes rightClickListener;

    @Override
    public void notifyValuesChanged() {
        this.repaint();
    }

    /**
     * Creates a new graph drawing panel.
     * 
     * @param network
     *            The network to display
     * @param leftClickListener
     *            An object that is notified if the user clicks on a node
     * @param rightClickListener
     *            An object that is notified if the user right clicks on a node
     */
    public PanelGraphDrawer(RegulatoryNetwork network, HasActivatableNodes leftClickListener, HasActivatableNodes rightClickListener) {
        if (network == null || leftClickListener == null || rightClickListener == null) {
            throw new NullPointerException();
        }

        this.network = network;
        this.leftClickListener = leftClickListener;
        this.rightClickListener = rightClickListener;
        network.addObserver(this);
        notifyNetworkChanged(); // Redraw background

        setBackground(Color.white);

        // Listen to left clicks and notify the leftClickListener
        addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                // We have to transform the physical location of the click to the virtual rectangle of the node
                Point clickPoint = e.getPoint();
                try {
                    backgroundGraphics.getTransform().inverseTransform(clickPoint, clickPoint);
                } catch (NoninvertibleTransformException e1) {
                    // This should not be possible with the transformations used in this window
                    e1.printStackTrace();
                }

                // Now search for the node under the click and discern the mouse buttons
                for (int i = 0; i < PanelGraphDrawer.this.network.size(); i++) {
                    NetworkNode node = PanelGraphDrawer.this.network.getNetworkNodes()[i];

                    Rectangle2D.Double nodeRect = node.getRectangle();

                    if (nodeRect.contains(clickPoint)) {
                        if (e.getButton() == MouseEvent.BUTTON1) {
                            PanelGraphDrawer.this.leftClickListener.activateNode(i);
                        }
                        if (e.getButton() == MouseEvent.BUTTON3) {
                            PanelGraphDrawer.this.rightClickListener.activateNode(i);
                        }
                        return;
                    }
                }
            }
        });
    }

    @Override
    public void notifyNetworkChanged() {
        // The network has changed, we have to redraw the lines on the background.

        if (network.size() == 0) {
            background = new BufferedImage(1, 1, BufferedImage.TYPE_4BYTE_ABGR);
            return;
        }

        // Determine dimensions

        Rectangle2D.Double dimensions = (Rectangle2D.Double) network.getNetworkNodes()[0].getRectangle().clone();

        // Choose a rectangle such that is encompasses all nodes, all lines and a border
        for (NetworkNode node : network.getNetworkNodes()) {
            dimensions.add(node.getRectangle());

            for (Connection connection : node.getConnections()) {
                for (Point2D.Double point : connection.getPath()) {
                    dimensions.add(point);
                }

                // The path origin and path source are included in the NODEPADDING
            }
        }

        double width = dimensions.getWidth() + 2 * NODEPADDING;
        double height = dimensions.getHeight() + 2 * NODEPADDING;

        // Create the new background set its transformation and its background color
        background = new BufferedImage((int) width, (int) height, BufferedImage.TYPE_4BYTE_ABGR);

        backgroundGraphics = (Graphics2D) background.getGraphics();
        backgroundGraphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        backgroundGraphics.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);

        backgroundGraphics.setColor(Color.white);
        backgroundGraphics.fillRect(0, 0, background.getWidth(), background.getHeight());

        backgroundGraphics.translate(-(dimensions.getX() - NODEPADDING), -(dimensions.getY() - NODEPADDING));

        // Set stroke and color for the arrows
        backgroundGraphics.setStroke(new BasicStroke(1.5f));
        backgroundGraphics.setColor(new Color(0, 0, 0));

        // Draw the arrows
        for (NetworkNode node : network.getNetworkNodes()) {
            for (int i = 0; i < node.getConnections().length; i++) {
                Connection connection = node.getConnections()[i];
                boolean inhib = false;
                if (node.getFunction() instanceof ActivatorInhibitorFunction) {
                    inhib = !((ActivatorInhibitorFunction) node.getFunction()).getActivators()[i];
                }
                drawConnectionArrow(backgroundGraphics, network.getNetworkNodes()[connection.getSource()], node, connection, inhib);
            }
        }

        // Set the new size of the panel and propagate the size.
        setPreferredSize(new Dimension(background.getWidth(), background.getHeight()));
        this.revalidate();

        // Draw the nodes.
        notifyValuesChanged();
    }

    @Override
    public void paintComponent(Graphics g) {
        // The values of the nodes might have changed, we have to redraw the nodes.

        // This also whitens the whole panel
        super.paintComponent(g);

        Graphics2D gg = (Graphics2D) g;

        // Copy the constant background to the panel.
        gg.drawImage(background, 0, 0, null);

        // Set rendering hints to use anti aliasing and high quality rendering
        gg.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        gg.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);

        // Add the transformation of the background BEFORE the current transformations of the Graphics2D (which are set by java)
        AffineTransform transform = new AffineTransform(backgroundGraphics.getTransform());
        transform.concatenate(gg.getTransform());
        gg.setTransform(transform);

        // Draw the nodes.
        gg.setStroke(new BasicStroke(2F));

        for (NetworkNode node : network.getNetworkNodes()) {
            Rectangle2D rect = node.getRectangle();
            RoundRectangle2D.Double roundedRect = new RoundRectangle2D.Double(rect.getX(), rect.getY(), rect.getWidth(), rect.getHeight(),
                    NODERECTANGLEROUNDING, NODERECTANGLEROUNDING);

            // Draw the background of the node
            float color = (float) (1 - node.getValue() * 0.5);
            gg.setColor(new Color(color, color, color));
            gg.fill(roundedRect);

            // Draw the border of the node
            gg.setColor(new Color(0, 0, 0));
            gg.draw(roundedRect);

            // Draw the name of the node
            FontMetrics metrics = gg.getFontMetrics();
            double labelWidth = metrics.stringWidth(node.getName());
            double labelHeight = metrics.getHeight();
            gg.setColor(new Color(0, 0, 0));
            gg.drawString(node.getName(), (float) (rect.getX() + rect.getWidth() / 2 - labelWidth / 2),
                    (float) (rect.getY() + rect.getHeight() / 2 + labelHeight / 2 - 3));
        }
    }

    /**
     * Draws an input arrow on the given Graphics2D. The parameters are taken from static fields and set in the Graphics2D (e.g. color and
     * stroke).
     * 
     * @param g
     *            The Graphics2D to draw on
     * @param source
     *            The source node
     * @param target
     *            The target node
     * @param connection
     *            The connection arrow to draw
     * @param inhibitory
     *            Whether the arrow is inhibitory
     */
    private void drawConnectionArrow(Graphics2D g, NetworkNode source, NetworkNode target, Connection connection, boolean inhibitory) {
        // Determine the first point of the input arrow; It doesn't matter if this is not on the border of the rectangle of the source node
        // since any superfluous line will just be overridden
        Path2D.Double arrowLine = new Path2D.Double();
        Point2D.Double firstPoint = MathLib
                .addPointsUnchecked(MathLib.getRectCenterUnchecked(source.getRectangle()), connection.getPathOrigin());
        arrowLine.moveTo(firstPoint.getX(), firstPoint.getY());

        // Draw all arrow segments except for the last and determine the source point of the last segment
        Point2D.Double s = null;

        if (connection.getPath().length != 0) {
            for (Point2D.Double point : connection.getPath()) {
                arrowLine.lineTo(point.getX(), point.getY());
                s = point;
            }

        } else {
            s = firstPoint;
        }

        g.draw(arrowLine);

        // Determine the target point of the last segment
        Point2D.Double t = MathLib.lineIntersectRectUnchecked(target.getRectangle(), s,
                MathLib.addPointsUnchecked(MathLib.getRectCenterUnchecked(target.getRectangle()), connection.getPathTarget()));

        // Draw the last segments as an arrow
        drawArrow(g, s, t, inhibitory);

    }

    /**
     * Draw an arrow on a Graphics2D. The parameters are taken from static fields and set in the Graphics2D (e.g. color and stroke).
     * 
     * @param g
     *            Graphics2D to draw on
     * @param source
     *            Source point
     * @param target
     *            Target point
     * @param inhibitory
     *            Whether the arrow is inhibitory
     */
    private void drawArrow(Graphics2D g, Point2D.Double source, Point2D.Double target, boolean inhibitory) {
        // Calculate the angle of the line from source to target
        double relx = target.getX() - source.getX();
        double rely = target.getY() - source.getY();

        double arrowAngle = Math.atan2(rely, relx);

        if (!inhibitory) {
            // Calculate the two points needed for the triangle at the tip of the arrow (in addition to the tip itself)
            double arrowtipangle = ARROWTIPANGLE;

            double x1 = -Math.cos(arrowAngle + arrowtipangle) * ARROWTIPLENGTH + target.getX();
            double y1 = -Math.sin(arrowAngle + arrowtipangle) * ARROWTIPLENGTH + target.getY();

            double x2 = -Math.cos(arrowAngle - arrowtipangle) * ARROWTIPLENGTH + target.getX();
            double y2 = -Math.sin(arrowAngle - arrowtipangle) * ARROWTIPLENGTH + target.getY();

            // Create the triagle and draw it
            Path2D.Double triangle = new Path2D.Double();
            triangle.moveTo(target.getX(), target.getY());
            triangle.lineTo(x1, y1);
            triangle.lineTo(x2, y2);
            g.fill(triangle);
        } else {
            double x1 = -Math.cos(arrowAngle + Math.PI / 2) * NEGATINGARROWWIDTH + target.getX();
            double y1 = -Math.sin(arrowAngle + Math.PI / 2) * NEGATINGARROWWIDTH + target.getY();

            double x2 = -Math.cos(arrowAngle - Math.PI / 2) * NEGATINGARROWWIDTH + target.getX();
            double y2 = -Math.sin(arrowAngle - Math.PI / 2) * NEGATINGARROWWIDTH + target.getY();

            double x3 = Math.cos(arrowAngle + Math.PI) * NEGATINGARROWTHICKNESS + x1;
            double y3 = Math.sin(arrowAngle + Math.PI) * NEGATINGARROWTHICKNESS + y1;

            double x4 = Math.cos(arrowAngle + Math.PI) * NEGATINGARROWTHICKNESS + x2;
            double y4 = Math.sin(arrowAngle + Math.PI) * NEGATINGARROWTHICKNESS + y2;

            Path2D.Double triangle = new Path2D.Double();
            triangle.moveTo(x1, y1);
            triangle.lineTo(x2, y2);
            triangle.lineTo(x4, y4);
            triangle.lineTo(x3, y3);

            g.fill(triangle);
        }

        // Draw the line of the arrow
        g.draw(new Line2D.Double(source, target));

    }

}
