package jimena.libs;

import java.io.Serializable;

/**
 * Encapsulates a double value. The object can be used in by-reference-calls and can be member of multiple arrays.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class DoubleValue implements Serializable {
    private static final long serialVersionUID = 929173770778383082L;
    private double value;
    private double min;
    private double max;

    /**
     * Creates a new double value.
     * 
     * @param value
     *            The initial value
     */
    public DoubleValue(double value) {
        // The value is checked by the other constructor
        this(value, -Double.MAX_VALUE, Double.MAX_VALUE);
    }

    /**
     * Creates a new double value whose value will alwas be between a given minimum and a given maximum.
     * 
     * @param value
     *            The initial value
     * @param min
     *            The minimum value
     * @param max
     *            The maximum value
     */
    public DoubleValue(double value, double min, double max) {
        // The value is checked by setValue
        this.min = min;
        this.max = max;
        setValue(value);
    }

    /**
     * Returns the value.
     * 
     * @return Current value
     */
    public double getValue() {
        return value;
    }

    /**
     * Sets the value.
     * 
     * @param value
     *            New value
     */
    public void setValue(Double value) {
        this.value = MathLib.limitToRangeAndAssertNumerical(value, min, max); // Also checks for NaN
    }

    /**
     * Add another double value to this value.
     * 
     * @param addend
     */
    public void addValue(Double addend) {
        setValue(value + addend);
    }

    @Override
    public String toString() {
        return StandardNumberFormat.NUMBERFORMAT.format(value);
    }
    
    
    @Override
    public DoubleValue clone(){
        return new DoubleValue(value, min, max);
    }
}
