package jimena.libs;

import java.util.List;
import java.util.Set;

/**
 * A library with functions for sets of SSS.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class SSSLib {

    /**
     * Returns the mean of the difference of set1 with set2 and the similarity of set2 with set1.
     * 
     * @param set1
     *            The set of arrays whose counterparts are to be searched in set2.
     * @param set2
     *            Where the counterparts are searched in.
     * @param componentsToConsider
     *            The vector component to consider
     * @return Mean of the difference of set1 with set2 and the similarity of set2 with set1.
     */
    public static double setDifference(List<double[]> set1, List<double[]> set2, Set<Integer> componentsToConsider) {
        if (set1.size() == 0 || set2.size() == 0) {
            return 1;
            // throw new IllegalArgumentException("The difference is not defined if one of the sets has no members.");
        }
        for (double[] s : set1) {
            if (set1.get(0).length != s.length) {
                throw new IllegalArgumentException("All members of both sets have to have the same size.");
            }
        }
        for (double[] s : set1) {
            if (set2.get(0).length != s.length) {
                throw new IllegalArgumentException("All members of both sets have to have the same size.");
            }
        }

        return (setDifferenceUnidirectionalUnchecked(set1, set2, componentsToConsider) + setDifferenceUnidirectionalUnchecked(set2, set1,
                componentsToConsider)) / 2;
    }

    /**
     * Returns the mean of the componentwise difference of set1 with set2 and the similarity of set2 with set1.
     * 
     * @param set1
     *            The set of arrays whose counterparts are to be searched in set2.
     * @param set2
     *            Where the counterparts are searched in.
     * @return Mean of the componentwise difference of set1 with set2 and the similarity of set2 with set1.
     */
    public static double[] setDifferenceSingle(List<double[]> set1, List<double[]> set2) {
        // TODO leere SSS arrays
        if (set1.size() == 0 || set2.size() == 0) {
            throw new IllegalArgumentException("The difference is not defined if one of the sets has no members.");
        }
        for (double[] s : set1) {
            if (set1.get(0).length != s.length) {
                throw new IllegalArgumentException("All members of both sets have to have the same size.");
            }
        }
        for (double[] s : set1) {
            if (set2.get(0).length != s.length) {
                throw new IllegalArgumentException("All members of both sets have to have the same size.");
            }
        }

        double[] result = setDifferenceUnidirectionalSingleUnchecked(set1, set2);
        MathLib.addDoubleArraysUnchecked(result, setDifferenceUnidirectionalSingleUnchecked(set2, set1));
        MathLib.divideDoubleArraysUnchecked(result, 2);

        return result;
    }

    /**
     * Returns the mean of the difference of set1 with set2 and the similarity of set2 with set1.
     * 
     * @param set1
     *            The set of arrays whose counterparts are to be searched in set2.
     * @param set2
     *            Where the counterparts are searched in.
     * @param componentsToConsider
     *            The vector component to consider
     * @return Mean of the difference of set1 with set2 and the similarity of set2 with set1.
     */
    public static double byteSetDifference(List<byte[]> set1, List<byte[]> set2, Set<Integer> componentsToConsider) {
        return setDifference(MathLib.byteArrayListToDoubleArrayListUnchecked(set1), MathLib.byteArrayListToDoubleArrayListUnchecked(set2),
                componentsToConsider);
    }

    /**
     * Returns the mean of the difference of the components of set1 with set2 and the similarity of set2 with set1.
     * 
     * @param set1
     *            The set of arrays whose counterparts are to be searched in set2.
     * @param set2
     *            Where the counterparts are searched in.
     * @return Mean of the difference of the components of set1 with set2 and the similarity of set2 with set1.
     */
    public static double[] byteSetDifferenceSingle(List<byte[]> set1, List<byte[]> set2) {
        return setDifferenceSingle(MathLib.byteArrayListToDoubleArrayListUnchecked(set1),
                MathLib.byteArrayListToDoubleArrayListUnchecked(set2));
    }

    /**
     * Returns the mean mean squared difference of the elements of set1 and their most similar counterparts in set2.
     * 
     * @param set1
     *            The set of arrays whose counterparts are to be searched in set2.
     * @param set2
     *            Where the counterparts are searched in.
     * @param componentsToConsider
     *            The vector component to consider
     * @return Mean mean squared difference of the elements of set1 and their most similar counterparts in set2
     */
    private static double setDifferenceUnidirectionalUnchecked(List<double[]> set1, List<double[]> set2, Set<Integer> componentsToConsider) {
        double result = 0;
        for (double[] member : set1) {
            result += mostSimilarDifferenceUnchecked(set2, member, componentsToConsider);
        }
        return result / set1.size();
    }

    /**
     * Returns the mean squared difference of the components of the elements of set1 and their most similar counterparts in set2.
     * 
     * @param set1
     *            The set of arrays whose counterparts are to be searched in set2.
     * @param set2
     *            Where the counterparts are searched in.
     * @return Mean squared difference of the components of the elements of set1 and their most similar counterparts in set2.
     */
    @SuppressWarnings("unchecked")
    private static double[] setDifferenceUnidirectionalSingleUnchecked(List<double[]> set1, List<double[]> set2) {
        // TODO: Definition für leere SSS Arrays
        double[] anElement;
        anElement = MathLib.randomElement(set1, set2);

        if (set1.size() == 0) {
            // TODO: temporary.
            return null;
        }

        double[] result = new double[anElement.length];

        for (double[] member : set1) {
            MathLib.addDoubleArraysUnchecked(result, mostSimilarDifferenceUncheckedSingle(set2, member));
        }
        MathLib.divideDoubleArraysUnchecked(result, set1.size());
        return result;
    }

    /**
     * Returns the mean squared difference of the most similar member of a set to a given needle.
     * 
     * @param haystack
     *            The set to search in.
     * @param needle
     *            The member whose most similar counterpart is to be searched.
     * @param componentsToConsider
     *            The vector component to consider
     * @return Mean squared difference of the most similar element
     */
    private static double mostSimilarDifferenceUnchecked(List<double[]> haystack, double[] needle, Set<Integer> componentsToConsider) {
        double minDifference = Double.MAX_VALUE;

        for (double[] candidate : haystack) {
            double difference = MathLib.meanSquaredDifferenceUnchecked(candidate, needle, componentsToConsider);
            if (minDifference > difference) {
                minDifference = difference;
            }
        }

        return minDifference;
    }

    /**
     * Returns the squared difference of the components the most similar member of a set to a given needle.
     * 
     * @param haystack
     *            The set to search in.
     * @param needle
     *            The member whose most similar counterpart is to be searched.
     * @return Squared difference of the components the most similar member of a set to a given needle.
     */
    private static double[] mostSimilarDifferenceUncheckedSingle(List<double[]> haystack, double[] needle) {
        double minDifference = Double.MAX_VALUE;
        double[] result = new double[needle.length];

        for (double[] candidate : haystack) {
            double difference = MathLib.meanSquaredDifferenceUnchecked(candidate, needle);
            if (minDifference > MathLib.meanSquaredDifferenceUnchecked(candidate, needle)) {
                minDifference = difference;
                result = MathLib.squaredDifferenceUnchecked(candidate, needle);
            }
        }

        return result;
    }

}
