package jimena.libs;

import java.awt.Component;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileFilter;

/**
 * A file chooser to save to txt files. Also provides static method as a function wrapper around itself.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class TxtFileChooser extends JFileChooser {
    private static final long serialVersionUID = -7783911168522977256L;
    private static FileFilter txtFilter = new FileFilter() {
        @Override
        public boolean accept(File file) {
            return file.getName().matches(".*\\.txt$") || file.isDirectory();
        }

        @Override
        public String getDescription() {
            return "txt-files (*.txt)";
        }
    };

    /**
     * Creates a new file chooser for txt files.
     */
    public TxtFileChooser() {
        for (FileFilter filter : getChoosableFileFilters()) {
            removeChoosableFileFilter(filter);
        }

        addChoosableFileFilter(txtFilter);

        setDialogTitle("Save to a text-file");
        setDialogType(JFileChooser.SAVE_DIALOG);
    }

    /**
     * Returns the txt file filter used by the file chooser.
     * 
     * @return The file filter
     */
    public static FileFilter getTxtFilter() {
        return txtFilter;
    }

    /**
     * Saves the given string to txt file that may be chosen with a new instance of this file chooser.
     * 
     * @param string
     *            The string to save.
     * @param parent
     *            The parent component of the window
     */
    public static void writeToTextFile(String string, Component parent) {
        JFileChooser filechooser = new TxtFileChooser();

        int returnVal = filechooser.showSaveDialog(parent);

        if (returnVal == JFileChooser.APPROVE_OPTION) {
            try {
                File file = filechooser.getSelectedFile();
                if (!TxtFileChooser.getTxtFilter().accept(filechooser.getSelectedFile())) {
                    file = new File(file.toString() + ".txt");
                }

                BufferedWriter writer = new BufferedWriter(new FileWriter(file));
                writer.write(string);
                writer.close();

            } catch (IOException ex) {
                JOptionPane.showMessageDialog(null, ex.getMessage(), "Export failed", JOptionPane.OK_OPTION ^ JOptionPane.ERROR_MESSAGE);
            }
        }
    }
}
