package jimena.perturbation;

import jimena.libs.DoubleValue;
import jimena.libs.MathLib;

/**
 * Implements a perturbation which is inactive before a given start and a takes a constant value after that until a given end.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class OnOffPerturbation extends RealValuesPerturbation {
    private DoubleValue start;
    private DoubleValue end;
    private DoubleValue value;
    private static final String[] valueNames = { "Start", "End", "Value" };

    /**
     * Creates a new perturbation which takes a constant value after an start until a given end.
     * 
     * @param start
     *            Start of the perturbation
     * @param end
     *            End of the perturbation
     * @param value
     *            Value of the perturbation
     */
    private OnOffPerturbation(DoubleValue start, DoubleValue end, DoubleValue value) {
        // Checks are done by the DoubleValues, the DoubleValues cannot be null since the function is private
        super(new DoubleValue[] { start, end, value }, valueNames);
        this.start = start;
        this.end = end;
        this.value = value;
    }

    /**
     * Creates a new perturbation which takes a constant value after an start until a given end.
     * 
     * If parameters are NaN, default values are chosen.
     * 
     * @param start
     *            Start of the perturbation
     * @param end
     *            End of the perturbation
     * @param value
     *            Value of the perturbation
     */
    public OnOffPerturbation(double start, double end, double value) {
        // Checks are done by the DoubleValues
        this(new DoubleValue(MathLib.isNotNaN(start) ? start : 0), new DoubleValue(MathLib.isNotNaN(end) ? end : 1000), new DoubleValue(
                MathLib.isNotNaN(value) ? value : 1, 0, 1));
    }

    /**
     * Creates a new perturbation which takes a constant value after until a given end.
     * 
     * If parameters are NaN, default values are chosen.
     * 
     * @param start
     *            start of the perturbation
     * @param value
     *            Value of the perturbation
     */
    public OnOffPerturbation(double start, double value) {
        // Checks are done by the DoubleValues
        this(start, Double.NaN, value);
    }

    /**
     * Creates a new perturbation which takes a constant value.
     * 
     * If parameters are NaN, default values are chosen.
     * 
     * @param value
     *            Value of the perturbation
     */
    public OnOffPerturbation(double value) {
        // Checks are done by the DoubleValues
        this(Double.NaN, Double.NaN, value);
    }

    /**
     * Creates a new perturbation which takes the constant value 1.
     */
    public OnOffPerturbation() {
        // Checks are done by the DoubleValues
        this(Double.NaN, Double.NaN, Double.NaN);
    }

    @Override
    public double getValue(double t) {
        return (t >= start.getValue() && t <= end.getValue()) ? value.getValue() : -1;
    }

    @Override
    public String getDescription() {
        return "Value " + value + " from " + start + " to " + end;
    }

    @Override
    public String toString() {
        return "On-Off-Perturbation";
    }

    @Override
    public Perturbation clone() {
        return new OnOffPerturbation(start.getValue(), end.getValue(), value.getValue());
    }

}
