package jimena.perturbation;


import java.util.TreeMap;

import javax.swing.table.AbstractTableModel;

import jimena.binaryrn.NetworkNode;
import jimena.binaryrn.RegulatoryNetwork;
import jimena.binaryrn.RegulatoryNetworkObserver;

/**
 * Implements a table model to display the perturbations of the network.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class PerturbationsModel extends AbstractTableModel implements RegulatoryNetworkObserver {
    private static final long serialVersionUID = -3638671347217871959L;
    private RegulatoryNetwork network;

    /**
     * Creates a new table models for the perturbations of the given network.
     * 
     * @param network
     *            The network whose perturbations are to be displayed
     */
    public PerturbationsModel(RegulatoryNetwork network) {
        this.network = network;

        network.addObserver(this);
        notifyNetworkChanged();
    }

    private TreeMap<Integer, Integer> mapRowToNode;
    private TreeMap<Integer, Integer> mapRowToPerturbation;

    @Override
    public int getColumnCount() {
        return 1;
    }

    @Override
    public int getRowCount() {
        return mapRowToNode.size();
    }

    @Override
    public Object getValueAt(int row, int col) {
        return nodeToPerturbation(row).getName() + ": " + rowToPerturbation(row).getDescription();
    }

    @Override
    public void notifyNetworkChanged() {
        notifyValuesChanged();
    }

    @Override
    public void notifyValuesChanged() {
        mapRowToNode = new TreeMap<Integer, Integer>();
        mapRowToPerturbation = new TreeMap<Integer, Integer>();

        int rowIndex = 0;
        int nodeIndex = 0;

        for (NetworkNode node : network.getNetworkNodes()) {
            int perturbationIndex = 0;
            for (@SuppressWarnings("unused")
            Perturbation perturbation : node.getPerturbations()) {
                mapRowToNode.put(rowIndex, nodeIndex);
                mapRowToPerturbation.put(rowIndex, perturbationIndex);
                rowIndex++;
                perturbationIndex++;
            }

            nodeIndex++;
        }

        fireTableStructureChanged();
    }

    @Override
    public Class<?> getColumnClass(int col) {
        return String.class;
    }

    @Override
    public boolean isCellEditable(int row, int col) {
        return false;
    }

    /**
     * Remove a perturbation from the network.
     * 
     * @param row
     *            Row of the perturbation in the table.
     */
    public void deletePerturbation(int row) {
        nodeToPerturbation(row).getPerturbations().remove((int) mapRowToPerturbation.get(row));

        network.notifyObserversOfChangedNetwork();
    }

    /**
     * Returns the perturbation for a given row.
     * 
     * @param row
     *            Row of the perturbation
     * @return The perturbation of the row
     */
    public Perturbation rowToPerturbation(int row) {
        return network.getNetworkNodes()[mapRowToNode.get(row)].getPerturbations().get((int) mapRowToPerturbation.get(row));
    }

    /**
     * Returns the node of the perturbation for a given row.
     * 
     * @param row
     *            Row of the perturbation
     * @return The node of the perturbation
     */
    public NetworkNode nodeToPerturbation(int row) {
        return network.getNetworkNodes()[mapRowToNode.get(row)];
    }
}
