package jimena.perturbation;

import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFormattedTextField;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.border.EmptyBorder;

import jimena.libs.DoubleValue;
import jimena.libs.StandardNumberFormat;

/**
 * Implements an abstract perturbations that is defined by real values.
 *
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 *
 */
public abstract class RealValuesPerturbation extends Perturbation {
    private DoubleValue[] values;
    private JFormattedTextField[] valueFields;
    private String[] valueNames;
    private boolean creationAborted = false;
    private JDialog window = null;

    @Override
    public JDialog getWindow() {
        if (window == null) {
            window = new RealValuesPerturbationWindow();
        }
        return window;
    }

    /**
     * A real values perturbation window.
     *
     * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
     *
     */
    private class RealValuesPerturbationWindow extends JDialog {
        private static final long serialVersionUID = 8361073243426465942L;

        /**
         * Creates a new real values perturbation window.
         */
        public RealValuesPerturbationWindow() {
            super((JFrame) null, "Perturbation settings", true);
            setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);

            valueFields = new JFormattedTextField[values.length];

            for (int i = 0; i < values.length; i++) {
                valueFields[i] = new JFormattedTextField(StandardNumberFormat.NUMBERFORMAT);
            }

            // Set up the window
            JPanel panel = new JPanel(new GridLayout(values.length + 1, 2, 12, 12));

            for (int i = 0; i < values.length; i++) {
                panel.add(new JLabel(valueNames[i]));
                panel.add(valueFields[i]);
            }

            JButton saveButton = new JButton("Save");
            saveButton.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    for (int i = 0; i < RealValuesPerturbation.this.values.length; i++) {
                        try {
                            RealValuesPerturbation.this.values[i].setValue(Double.valueOf(RealValuesPerturbation.this.valueFields[i]
                                    .getText()));
                        } catch (NumberFormatException n) {
                            // Do nothing
                        }
                    }

                    RealValuesPerturbationWindow.this.setVisible(false);
                }
            });
            panel.add(saveButton);
            this.getRootPane().setDefaultButton(saveButton);

            JButton abortButton = new JButton("Abort");
            abortButton.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    RealValuesPerturbation.this.creationAborted = true;
                    RealValuesPerturbationWindow.this.setVisible(false);
                }
            });
            panel.add(abortButton);
            panel.setBorder(new EmptyBorder(10, 10, 10, 10));

            setContentPane(panel);

            setMinimumSize(new Dimension(300, (values.length + 1) * 42));
            setResizable(false);

            addWindowListener(new WindowAdapter() {
                @Override
                public void windowOpened(WindowEvent arg0) {
                    for (int i = 0; i < RealValuesPerturbation.this.values.length; i++) {
                        RealValuesPerturbation.this.valueFields[i].setText(StandardNumberFormat.NUMBERFORMAT
                                .format(RealValuesPerturbation.this.values[i].getValue()));
                    }
                }
            });
        }
    }

    /**
     * Creates a new real values perturbation.
     *
     * @param values
     *            Values that define the perturbation
     * @param valueNames
     *            The names of the values
     */
    public RealValuesPerturbation(DoubleValue[] values, String[] valueNames) {
        super();

        if (values.length != valueNames.length) {
            throw new IllegalArgumentException("Number of values must correspond to the number of value Descriptions.");
        }

        for (DoubleValue value : values) {
            if (value == null) {
                throw new NullPointerException();
            }
        }

        for (String valueName : valueNames) {
            if (valueName == null) {
                throw new NullPointerException();
            }
        }

        this.values = values;
        this.valueNames = valueNames;
    }

    @Override
    public boolean creationWasAborted() {
        return creationAborted;
    }
}
