package jimena.settings;


import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;

import javax.swing.JTextField;

import jimena.libs.MathLib;

/**
 * Encapsulates Information about settings in this settings window.
 * 
 * This class represents a setting of type double.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class DoubleSetting extends AbstractSetting<Double> {
    /**
     * Creates a new Setting. Every setting will automatically be registered for the settings windows.
     * 
     * @param key
     *            Key of this setting a Preferences structure
     * @param label
     *            A short label of this setting
     * @param description
     *            The description of this setting, if null is given the label will be used
     * @param minValue
     *            The minimum value
     * @param maxValue
     *            The maximum value
     * @param defaultValue
     *            The default value
     */
    public DoubleSetting(String key, String label, String description, double minValue, double maxValue, double defaultValue) {
        super(key, label, description, minValue, maxValue, defaultValue);
        if (MathLib.isNaN(minValue) || MathLib.isNaN(maxValue)) {
            throw new IllegalArgumentException("No NaN values allowed.");
        }

        MathLib.checkNotNaNAndWithinRange(defaultValue, minValue, maxValue);
    }

    @Override
    public Double getValue() {
        return preferences.getDouble(getKey(), defaultValue);
    }

    /**
     * A text field bound to this setting.
     * 
     * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
     * 
     */
    public class DoubleSettingsTextField extends SettingsTextField {
        private static final long serialVersionUID = -8521178246566337503L;

        /**
         * Creates a new text field for the setting.
         */
        public DoubleSettingsTextField() {
            setText(String.valueOf(DoubleSetting.this.getValue()));
            addFocusListener(new FocusAdapter() {
                @Override
                public void focusLost(FocusEvent arg0) {
                    try {
                        if (Double.valueOf(DoubleSettingsTextField.this.getText()) < minValue) {
                            DoubleSettingsTextField.this.setText(String.valueOf(minValue));
                        }
                        if (Double.valueOf(DoubleSettingsTextField.this.getText()) > maxValue) {
                            DoubleSettingsTextField.this.setText(String.valueOf(maxValue));
                        }

                    } catch (NumberFormatException e) {
                        DoubleSettingsTextField.this.setText(String.valueOf(defaultValue));
                    }

                    preferences.putDouble(getKey(), Double.valueOf(DoubleSettingsTextField.this.getText()));
                }
            });
        }

        @Override
        public void reset() {
            DoubleSettingsTextField.this.setText(String.valueOf(defaultValue));
            preferences.putDouble(getKey(), Double.valueOf(DoubleSettingsTextField.this.getText()));
        }
    }

    private SettingsTextField settingsTextField = new DoubleSettingsTextField();

    @Override
    public JTextField getTextField() {
        return settingsTextField;
    }

    @Override
    public void reset() {
        settingsTextField.reset();
    }

}
