package jimena.settings;



import java.awt.Dimension;
import java.util.prefs.Preferences;

import javax.swing.JTextField;

import jimena.gui.guilib.AutoselectingTextField;

/**
 * Encapsulates Information about settings in this settings window.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public abstract class RawSetting {
    /**
     * Dimension of the text field generated in the getTextField method.
     */
    static protected final Dimension textFieldDimension = new Dimension(100, 20);

    private String key;
    private String label;
    private String description;

    /**
     * A preferences structure for persistent storing
     */
    protected static final Preferences preferences = Preferences.userNodeForPackage(SettingsWindow.class);

    /**
     * Creates a new Setting. Every setting will automatically be registered for the settings windows.
     * 
     * @param key
     *            Key of this setting a Preferences structure
     * @param label
     *            A short label of this setting
     * @param description
     *            The description of this setting, if null is given the label will be used
     */
    public RawSetting(String key, String label, String description) {
        if (key == null || label == null) {
            throw new NullPointerException();
        }

        if (description == null) {
            description = label;
        }

        this.key = key;
        this.label = label;
        this.description = description;

        SettingsWindow.registerSetting(this);
    }

    /**
     * Returns a TextField for the setting.
     * 
     * @return A TextField that is bound to the setting.
     */
    public abstract JTextField getTextField();

    /**
     * Returns the key of this setting a Preferences structure.
     * 
     * @return The key
     */
    public String getKey() {
        return key;
    }

    /**
     * Returns a short label of this setting.
     * 
     * @return Label of this setting
     */
    public String getLabel() {
        return label;
    }

    /**
     * Returns the description of this setting.
     * 
     * @return Description of this setting
     */
    public String getDescription() {
        return description;
    }

    /**
     * Resets the setting (this includes its TextField)
     */
    public abstract void reset();

    /**
     * A common superclass for all text field representing a setting.
     * 
     * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
     * 
     */
    protected abstract class SettingsTextField extends AutoselectingTextField {
        private static final long serialVersionUID = 4945361348848721210L;

        /**
         * Performs calls common to all text fields representing a setting.
         */
        public SettingsTextField() {
            setPreferredSize(textFieldDimension);
            setToolTipText(getDescription());
        }

        /**
         * Resets the TextField and the setting.
         */
        public abstract void reset();
    }
}
