package jimena.settings;

import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.ArrayList;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.border.EmptyBorder;

/**
 * Implements a windows to set the settings of the application.
 *
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 *
 */
public class SettingsWindow extends JFrame {
    private static final long serialVersionUID = 371183139603442630L;

    private static ArrayList<RawSetting> settings = new ArrayList<RawSetting>();

    /**
     * Adds a setting the the settings windows.
     *
     * @param setting
     *            The setting to add
     */
    protected static void registerSetting(RawSetting setting) {
        settings.add(setting);
    }

    /**
     * Maximum difference between duplicates.
     */
    public final static DoubleSetting MAXDUPLICATEDIFFERENCE = new DoubleSetting("maxduplicatediff",
            "Maximum difference between duplicates", null, 0.00001, 0.1, 0.002);
    /**
     * Maximum deviation for a curve to be considered stable.
     */
    public final static DoubleSetting MAXSTABILITYDIFFERENCE = new DoubleSetting("maxstabilitydiff",
            "Maximum deviation for a curve to be considered stable", null, 0.00001, 0.1, 0.001);
    /**
     * Minimum time of the curve within the maximum deviation.
     */
    public final static DoubleSetting MINSTABILITYTIME = new DoubleSetting("minstabilitytime", "Minimum time within the maximum deviation",
            null, 0.1, 1000, 5);

    /**
     * Minimum (simulation) time between two log entries during the simulation.
     */
    public final static DoubleSetting MINSIMULATIONTIMEBETWEENLOGS = new DoubleSetting("minsimulationtimebetweenlogs",
            "Minimum (simulation) time between two log entries during the simulation", null, 0.00001, 1, 0.001);

    /**
     * Maximum number of threads used in multithreaded calculations.
     */
    public final static IntegerSetting THREADS = new IntegerSetting("threads",
            "Maximum number of threads used in multithreaded calculations", null, 1, 256, Runtime.getRuntime().availableProcessors());

    /**
     * Creates a new settings window.
     */
    public SettingsWindow() {
        // Set up the window
        setIconImage(new ImageIcon("images" + File.separator + "settings16.png").getImage());
        setTitle("Settings");
        getContentPane().setLayout(new BorderLayout());

        // Create the grid with the labels and inputs boxes of the settings
        JPanel settingsPanel = new JPanel();
        settingsPanel.setLayout(new GridLayout(settings.size(), 1, 12, 12));
        getContentPane().add(settingsPanel, BorderLayout.CENTER);

        for (RawSetting setting : settings) {
            JPanel settingPanel = new JPanel(new BorderLayout(10, 0));
            settingPanel.add(new JLabel(setting.getLabel()), BorderLayout.CENTER);
            settingPanel.add(setting.getTextField(), BorderLayout.EAST);
            settingsPanel.add(settingPanel);
        }

        settingsPanel.setBorder(new EmptyBorder(10, 10, 10, 10));

        // Create a panel with control buttons at the bottom of the window
        JPanel buttonsPanel = new JPanel();
        buttonsPanel.setLayout(new GridLayout(1, 2));

        JButton closeWindow = new JButton("Reset All");
        closeWindow.setToolTipText("You can save a value by clicking on another input box. "
                + "Enter a invalid string to reset an input box to the default value.");
        closeWindow.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                for (RawSetting setting : settings) {
                    setting.reset();
                }
            }
        });

        buttonsPanel.add(closeWindow);
        getContentPane().add(buttonsPanel, BorderLayout.SOUTH);

        setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
        setResizable(false);
        pack();
    }
}
