package jimena.simulation;

import jimena.binaryrn.RegulatoryNetwork;
import jimena.calculationparameters.SimulationParameters;
import jimena.libs.ChecksLib;
import jimena.simulationmethods.SimulationMethod;

/**
 * Implements a simulation of the network in a separate thread.
 *
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 *
 */
public class Simulator extends Thread {
    private double minSimulationTimeBetweenLogs;
    private static final double minTimeBetweenNotifications = 50;
    private RegulatoryNetwork network;
    private SimulationMethod method;
    private double dt;
    private double maxt;
    private double maxSpeed;
    private CalculationController calculationController;

    /**
     * Creates a new Simulator.
     *
     * @param parameters
     *            Parameters for the simulation
     * @param network
     *            The network to simulate
     * @param maxSpeed
     *            Maximum speed of the simulation
     * @param minSimulationTimeBetweenLogs
     *            Minimum (simulation) time between two log entries during the simulation.
     */
    public Simulator(SimulationParameters parameters, RegulatoryNetwork network, double maxSpeed, double minSimulationTimeBetweenLogs) {
        ChecksLib.checkNotNull(network, parameters.getMethod());
        if (parameters.getDt() <= 0 || parameters.getMaxT() < 0 || maxSpeed <= 0 || minSimulationTimeBetweenLogs <= 0
                || minTimeBetweenNotifications <= 0) {
            throw new IllegalArgumentException("All numerical inputs to the constuctor of a Simulator must positive.");
        }

        this.network = network;
        this.method = parameters.getMethod();
        this.dt = parameters.getDt();
        this.maxt = parameters.getMaxT();
        this.maxSpeed = maxSpeed;
        this.minSimulationTimeBetweenLogs = minSimulationTimeBetweenLogs;
        this.calculationController = parameters.getCalculationController();
    }

    /**
     * Contains the main code of the simulation.
     *
     * Do NOT call this method, but use start() instead;
     *
     */
    @Override
    public void run() {
        if (network.size() == 0) {

        } else {
            double t = 0; // This is a different t than in the simulation
            long lastNotification = System.currentTimeMillis();
            int stepsDone = 0;
            int stepsPerLap = (int) ((maxSpeed * minTimeBetweenNotifications) / dt / 1000);
            int stepsDoneAtLastNotification = 0;
            double timeIndexAtLastLog = 0;

            while ((t + dt) < (maxt + dt / 10) && (calculationController != null ? calculationController.isOn() : true)) {
                if ((stepsDone - stepsDoneAtLastNotification) <= stepsPerLap) {
                    // We haven't calculated enough steps for the next notification of the observers
                    method.stepUpdateNetwork(network, dt);

                    t += dt;
                    stepsDone++;
                } else {
                    // We have calculated enough steps for the next notification of the observers
                    try {
                        Thread.sleep(1);
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                }

                // Update the observers of the network every minTimeBetweenNotifications milliseconds
                if ((System.currentTimeMillis() - lastNotification) >= minTimeBetweenNotifications) {
                    // Delay the simulation (and the delivery of the results) according to the maximum simulation speed.
                    stepsDoneAtLastNotification = stepsDone;

                    if (calculationController != null) {
                        calculationController.setProgress(t / maxt, stepsDone);
                    }

                    network.notifyObserversOfChangedValues();
                    lastNotification = System.currentTimeMillis();
                }

                // Update the log every MINSIMULATIONTIMEBETWEENLOGS
                if ((network.getTimeIndex().getValue() - timeIndexAtLastLog) >= minSimulationTimeBetweenLogs) {
                    network.log();
                    timeIndexAtLastLog = network.getTimeIndex().getValue();
                }
            }

            network.notifyObserversOfChangedValues();
            if (calculationController != null) {
                calculationController.notifyCalculationFinished();
            }
        }
    }
}
