package jimena.simulation;

import jimena.binaryrn.RegulatoryNetwork;
import jimena.calculationparameters.ConvergenceParameters;
import jimena.libs.ChecksLib;
import jimena.simulationmethods.SimulationMethod;

/**
 * Implements a searching of stable steady states from a given set of start values in a separate thread.
 * 
 * @author Stefan Karl, Department of Bioinformatics, University of Würzburg, stefan[dot]karl[at]uni-wuerzburg[dot]de
 * 
 */
public class StableSteadyState extends ConvergenceFromStartVector {
    private RegulatoryNetwork network;
    private SimulationMethod method;
    private double dt;
    private double maxt;
    private double stabilityMaxDiff;
    private double stabilityMinTime;
    private CalculationController calculationController = null;
    private static final long minTimeBetweenNotifications = 50;

    /**
     * Creates a new stable steady state searcher.
     * 
     * @param network
     *            The network to search stable steady states in
     * @param p
     *            Simulation parameters
     */
    public StableSteadyState(RegulatoryNetwork network, ConvergenceParameters p) {
        this(network, p.getMethod(), p.getDt(), p.getMaxT(), p.getStabilityMaxDiff(), p.getStabilityMinTime(), p.getCalculationController());
    }

    /**
     * Creates a new stable steady state searcher.
     * 
     * @param network
     *            The network to search stable steady states in
     * @param method
     *            The method to apply for the simulation
     * @param dt
     *            The time step in the simulation
     * @param maxt
     *            The maximum simulated time
     * @param stabilityMaxDiff
     *            The epsilon neighborhood around a former value which must not be left to assume stability has the diameter
     *            stabilityMaxDiff*2
     * @param stabilityMinTime
     *            Time the node has to spend in an epsilon neighborhood around a former value to assume stability
     * @param calculationController
     *            A calculation controller or null for an uncontrolled calculation
     */
    public StableSteadyState(RegulatoryNetwork network, SimulationMethod method, double dt, double maxt, double stabilityMaxDiff,
            double stabilityMinTime, CalculationController calculationController) {
        ChecksLib.checkNotNull(network, method);

        if (dt <= 0 || maxt <= 0 || stabilityMaxDiff <= 0 || stabilityMinTime <= 0) {
            throw new IllegalArgumentException("All numerical inputs to the constuctor of a StableSteadyState must positive.");
        }

        this.network = network;
        this.method = method;
        this.dt = dt;
        this.maxt = maxt;
        this.stabilityMaxDiff = stabilityMaxDiff;
        this.stabilityMinTime = stabilityMinTime;

        this.calculationController = calculationController;
    }

    @Override
    public ConvergenceResult call() throws Exception {
        double t = 0;
        if (network.size() == 0) {
            return null;
        } else {
            double lastStabilityCheckPoint = 0;
            int stepsDone = 0;
            long lastNotification = System.currentTimeMillis();
            double[] valuesAtLastStabilityCheckPoint = network.getValues();
            boolean StabilityCheckFailed = false;

            while (t <= maxt && (calculationController != null ? calculationController.isOn() : true)) {
                // Simulate the network
                method.stepUpdateNetwork(network, dt);

                // Do not check the stability if the check has already failed once during the current interval
                if (!StabilityCheckFailed) {
                    for (int i = 0; i < network.size(); i++) {
                        if (Math.abs(network.getNetworkNodes()[i].getValue() - valuesAtLastStabilityCheckPoint[i]) > stabilityMaxDiff) {
                            StabilityCheckFailed = true;
                        }
                    }
                }

                // Abort at the end of the interval if all stability checks where positive
                if ((t - lastStabilityCheckPoint) >= stabilityMinTime) {
                    if (!StabilityCheckFailed) {
                        return new ConvergenceResult(network.getValues(), t);
                    }

                    valuesAtLastStabilityCheckPoint = network.getValues().clone();
                    StabilityCheckFailed = false;
                    lastStabilityCheckPoint = t;
                }

                t += dt;
                stepsDone++;

                // Update the progress windows if there is one
                if (calculationController != null && (System.currentTimeMillis() - lastNotification) >= minTimeBetweenNotifications) {
                    calculationController.setProgress(t / maxt, stepsDone);
                    lastNotification = System.currentTimeMillis();
                }
            }

            if (calculationController != null) {
                calculationController.notifyCalculationFinished();
            }

            return null;
        }
    }
}
