/*
 * Decompiled with CFR 0.152.
 */
package org.opensourcephysics.numerics;

import java.util.ArrayList;
import org.opensourcephysics.numerics.Function;
import org.opensourcephysics.numerics.Root;
import org.opensourcephysics.numerics.Util;

public class Polynomial
implements Function {
    protected double[] coefficients;

    public Polynomial(double[] coef) {
        this.coefficients = coef;
    }

    public double[] getCoefficients() {
        return (double[])this.coefficients.clone();
    }

    public Polynomial(String[] coef) {
        this.coefficients = new double[coef.length];
        int n = coef.length;
        for (int i = 0; i < n; ++i) {
            try {
                this.coefficients[i] = Double.parseDouble(coef[i]);
                continue;
            }
            catch (NumberFormatException ex) {
                this.coefficients[i] = 0.0;
            }
        }
    }

    public static double evalPolynomial(double x, double[] coeff) {
        int n = coeff.length - 1;
        double y = coeff[n];
        for (int i = n - 1; i >= 0; --i) {
            y = coeff[i] + y * x;
        }
        return y;
    }

    public Polynomial add(double r) {
        int n = this.coefficients.length;
        double[] coef = new double[n];
        coef[0] = this.coefficients[0] + r;
        for (int i = 1; i < n; ++i) {
            coef[i] = this.coefficients[i];
        }
        return new Polynomial(coef);
    }

    public Polynomial add(Polynomial p) {
        int n = Math.max(p.degree(), this.degree()) + 1;
        double[] coef = new double[n];
        for (int i = 0; i < n; ++i) {
            coef[i] = this.coefficient(i) + p.coefficient(i);
        }
        return new Polynomial(coef);
    }

    public double coefficient(int n) {
        return n < this.coefficients.length ? this.coefficients[n] : 0.0;
    }

    public Polynomial deflate(double r) {
        int n = this.degree();
        double remainder = this.coefficients[n];
        double[] coef = new double[n];
        for (int k = n - 1; k >= 0; --k) {
            coef[k] = remainder;
            remainder = remainder * r + this.coefficients[k];
        }
        return new Polynomial(coef);
    }

    public int degree() {
        return this.coefficients.length - 1;
    }

    public Polynomial derivative() {
        int n = this.degree();
        if (n == 0) {
            double[] coef = new double[]{0.0};
            return new Polynomial(coef);
        }
        double[] coef = new double[n];
        for (int i = 1; i <= n; ++i) {
            coef[i - 1] = this.coefficients[i] * (double)i;
        }
        return new Polynomial(coef);
    }

    public Polynomial divide(double r) {
        return this.multiply(1.0 / r);
    }

    public Polynomial divide(Polynomial p) {
        return this.divideWithRemainder(p)[0];
    }

    public Polynomial[] divideWithRemainder(Polynomial p) {
        int n;
        Polynomial[] answer = new Polynomial[2];
        int m = this.degree();
        if (m < (n = p.degree())) {
            double[] q = new double[]{0.0};
            answer[0] = new Polynomial(q);
            answer[1] = p;
            return answer;
        }
        double[] quotient = new double[m - n + 1];
        double[] coef = new double[m + 1];
        for (int k = 0; k <= m; ++k) {
            coef[k] = this.coefficients[k];
        }
        double norm = 1.0 / p.coefficient(n);
        for (int k = m - n; k >= 0; --k) {
            quotient[k] = coef[n + k] * norm;
            for (int j = n + k - 1; j >= k; --j) {
                int n2 = j;
                coef[n2] = coef[n2] - quotient[k] * p.coefficient(j - k);
            }
        }
        double[] remainder = new double[n];
        for (int k = 0; k < n; ++k) {
            remainder[k] = coef[k];
        }
        answer[0] = new Polynomial(quotient);
        answer[1] = new Polynomial(remainder);
        return answer;
    }

    public Polynomial integral() {
        return this.integral(0.0);
    }

    public Polynomial integral(double value) {
        int n = this.coefficients.length + 1;
        double[] coef = new double[n];
        coef[0] = value;
        for (int i = 1; i < n; ++i) {
            coef[i] = this.coefficients[i - 1] / (double)i;
        }
        return new Polynomial(coef);
    }

    public Polynomial multiply(double r) {
        int n = this.coefficients.length;
        double[] coef = new double[n];
        for (int i = 0; i < n; ++i) {
            coef[i] = this.coefficients[i] * r;
        }
        return new Polynomial(coef);
    }

    public Polynomial multiply(Polynomial p) {
        int n = p.degree() + this.degree();
        double[] coef = new double[n + 1];
        for (int i = 0; i <= n; ++i) {
            coef[i] = 0.0;
            for (int k = 0; k <= i; ++k) {
                int n2 = i;
                coef[n2] = coef[n2] + p.coefficient(k) * this.coefficient(i - k);
            }
        }
        return new Polynomial(coef);
    }

    public double[] roots() {
        return this.roots(Util.defaultNumericalPrecision);
    }

    public double[] roots(double desiredPrecision) {
        double r;
        int counter;
        double start = 0.0;
        if (this.degree() < 1) {
            return new double[0];
        }
        Polynomial dp = this.derivative();
        for (counter = 0; counter < 100 && Math.abs(dp.evaluate(start)) < desiredPrecision; ++counter) {
            start = Math.random();
        }
        Polynomial p = this;
        ArrayList<Double> list = new ArrayList<Double>(this.degree());
        while (!Double.isNaN(r = Root.newton(p, dp, start, desiredPrecision))) {
            list.add(new Double(r));
            p = p.deflate(r);
            if (p.degree() == 0) break;
            dp = p.derivative();
            start = 0.0;
            for (counter = 0; counter < 100 && Math.abs(dp.evaluate(start)) < desiredPrecision; ++counter) {
                start = Math.random();
            }
        }
        double[] roots = new double[list.size()];
        int n = list.size();
        for (int i = 0; i < n; ++i) {
            roots[i] = (Double)list.get(i);
        }
        return roots;
    }

    public Polynomial subtract(double r) {
        return this.add(-r);
    }

    public Polynomial subtract(Polynomial p) {
        int n = Math.max(p.degree(), this.degree()) + 1;
        double[] coef = new double[n];
        for (int i = 0; i < n; ++i) {
            coef[i] = this.coefficient(i) - p.coefficient(i);
        }
        return new Polynomial(coef);
    }

    public String toString() {
        if (this.coefficients == null || this.coefficients.length < 1) {
            return "Polynomial coefficients are undefined.";
        }
        StringBuffer sb = new StringBuffer();
        boolean firstNonZeroCoefficientPrinted = false;
        int m = this.coefficients.length;
        for (int n = 0; n < m; ++n) {
            if (this.coefficients[n] == 0.0) continue;
            if (firstNonZeroCoefficientPrinted) {
                sb.append(this.coefficients[n] > 0.0 ? " + " : " ");
            } else {
                firstNonZeroCoefficientPrinted = true;
            }
            if (n == 0 || this.coefficients[n] != 1.0) {
                sb.append(Double.toString(this.coefficients[n]));
            }
            if (n <= 0) continue;
            sb.append(" x^" + n);
        }
        String str = sb.toString();
        if (str.equals("")) {
            return "0";
        }
        return str;
    }

    public double evaluate(double x) {
        int n = this.coefficients.length;
        double answer = this.coefficients[--n];
        while (n > 0) {
            answer = answer * x + this.coefficients[--n];
        }
        return answer;
    }

    public double[] valueAndDerivative(double x) {
        int n = this.coefficients.length;
        double[] answer = new double[]{this.coefficients[--n], 0.0};
        while (n > 0) {
            answer[1] = answer[1] * x + answer[0];
            answer[0] = answer[0] * x + this.coefficients[--n];
        }
        return answer;
    }
}

